package drr.base.trade.basket.functions;

import cdm.product.template.Basket;
import cdm.product.template.BasketConstituent;
import cdm.product.template.Product;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.MapperS;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(GetBasket.GetBasketDefault.class)
public abstract class GetBasket implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;

	/**
	* @param basket 
	* @return basketConstituent 
	*/
	public List<? extends Product> evaluate(Basket basket) {
		List<Product.ProductBuilder> basketConstituentBuilder = doEvaluate(basket);
		
		final List<? extends Product> basketConstituent;
		if (basketConstituentBuilder == null) {
			basketConstituent = null;
		} else {
			basketConstituent = basketConstituentBuilder.stream().map(Product::build).collect(Collectors.toList());
			objectValidator.validate(Product.class, basketConstituent);
		}
		
		return basketConstituent;
	}

	protected abstract List<Product.ProductBuilder> doEvaluate(Basket basket);

	public static class GetBasketDefault extends GetBasket {
		@Override
		protected List<Product.ProductBuilder> doEvaluate(Basket basket) {
			List<Product.ProductBuilder> basketConstituent = new ArrayList<>();
			return assignOutput(basketConstituent, basket);
		}
		
		protected List<Product.ProductBuilder> assignOutput(List<Product.ProductBuilder> basketConstituent, Basket basket) {
			if (exists(MapperS.of(basket).<Product>mapC("getBasketConstituent", _basket -> _basket.getBasketConstituent())).getOrDefault(false)) {
				basketConstituent = toBuilder(MapperS.of(basket).<Product>mapC("getBasketConstituent", _basket -> _basket.getBasketConstituent()).getMulti());
			} else {
				basketConstituent = toBuilder(MapperS.of(basket).<BasketConstituent>mapC("getPortfolioBasketConstituent", _basket -> _basket.getPortfolioBasketConstituent()).getMulti());
			}
			
			return Optional.ofNullable(basketConstituent)
				.map(o -> o.stream().map(i -> i.prune()).collect(Collectors.toList()))
				.orElse(null);
		}
	}
}
