package drr.regulation.common;

import cdm.event.common.TradeIdentifier;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import drr.regulation.common.meta.TransactionInformationMeta;
import drr.regulation.common.metafields.FieldWithMetaRegimeNameEnum;
import drr.regulation.common.metafields.FieldWithMetaSupervisoryBodyEnum;
import java.util.Objects;

import static java.util.Optional.ofNullable;

/**
 * @version 7.0.0-dev.66
 */
@RosettaDataType(value="TransactionInformation", builder=TransactionInformation.TransactionInformationBuilderImpl.class, version="7.0.0-dev.66")
@RuneDataType(value="TransactionInformation", model="drr", builder=TransactionInformation.TransactionInformationBuilderImpl.class, version="7.0.0-dev.66")
public interface TransactionInformation extends RosettaModelObject {

	TransactionInformationMeta metaData = new TransactionInformationMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 * Identifies the reporting regime under which this data is reported. For example, Dodd-Frank, MiFID, HongKongOTCDRepository, ODRF
	 */
	FieldWithMetaRegimeNameEnum getRegimeName();
	/**
	 * Identifies the specific regulator or other supervisory body for which this data is produced. For example, CFTC, SEC, UKFSA, ODRF, SFC, ESMA.
	 */
	FieldWithMetaSupervisoryBodyEnum getSupervisoryBody();
	/**
	 * Identifier to use specifically in the context of reporting to a specific regime
	 */
	TradeIdentifier getTransactionIdentifier();
	/**
	 * Identifier of the prior transaction
	 */
	TradeIdentifier getPriorTransactionIdentifier();
	/**
	 * Transaction information required for reporting a transaction under ESMA.
	 */
	ESMATransactionInformation getEsmaTransactionInformation();
	/**
	 * Transaction information required for reporting a transaction under FCA.
	 */
	CommonTransactionInformation getFcaTransactionInformation();
	/**
	 * Transaction information required for reporting a transaction under MAS.
	 */
	CommonTransactionInformation getMasTransactionInformation();
	/**
	 * Transaction information required for reporting a transaction under CSA.
	 */
	CommonTransactionInformation getCsaTransactionInformation();
	/**
	 * Transaction information required for reporting a transaction under HKMA.
	 */
	CommonTransactionInformation getHkmaTransactionInformation();

	/*********************** Build Methods  ***********************/
	TransactionInformation build();
	
	TransactionInformation.TransactionInformationBuilder toBuilder();
	
	static TransactionInformation.TransactionInformationBuilder builder() {
		return new TransactionInformation.TransactionInformationBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends TransactionInformation> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends TransactionInformation> getType() {
		return TransactionInformation.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processRosetta(path.newSubPath("regimeName"), processor, FieldWithMetaRegimeNameEnum.class, getRegimeName());
		processRosetta(path.newSubPath("supervisoryBody"), processor, FieldWithMetaSupervisoryBodyEnum.class, getSupervisoryBody());
		processRosetta(path.newSubPath("transactionIdentifier"), processor, TradeIdentifier.class, getTransactionIdentifier());
		processRosetta(path.newSubPath("priorTransactionIdentifier"), processor, TradeIdentifier.class, getPriorTransactionIdentifier());
		processRosetta(path.newSubPath("esmaTransactionInformation"), processor, ESMATransactionInformation.class, getEsmaTransactionInformation());
		processRosetta(path.newSubPath("fcaTransactionInformation"), processor, CommonTransactionInformation.class, getFcaTransactionInformation());
		processRosetta(path.newSubPath("masTransactionInformation"), processor, CommonTransactionInformation.class, getMasTransactionInformation());
		processRosetta(path.newSubPath("csaTransactionInformation"), processor, CommonTransactionInformation.class, getCsaTransactionInformation());
		processRosetta(path.newSubPath("hkmaTransactionInformation"), processor, CommonTransactionInformation.class, getHkmaTransactionInformation());
	}
	

	/*********************** Builder Interface  ***********************/
	interface TransactionInformationBuilder extends TransactionInformation, RosettaModelObjectBuilder {
		FieldWithMetaRegimeNameEnum.FieldWithMetaRegimeNameEnumBuilder getOrCreateRegimeName();
		@Override
		FieldWithMetaRegimeNameEnum.FieldWithMetaRegimeNameEnumBuilder getRegimeName();
		FieldWithMetaSupervisoryBodyEnum.FieldWithMetaSupervisoryBodyEnumBuilder getOrCreateSupervisoryBody();
		@Override
		FieldWithMetaSupervisoryBodyEnum.FieldWithMetaSupervisoryBodyEnumBuilder getSupervisoryBody();
		TradeIdentifier.TradeIdentifierBuilder getOrCreateTransactionIdentifier();
		@Override
		TradeIdentifier.TradeIdentifierBuilder getTransactionIdentifier();
		TradeIdentifier.TradeIdentifierBuilder getOrCreatePriorTransactionIdentifier();
		@Override
		TradeIdentifier.TradeIdentifierBuilder getPriorTransactionIdentifier();
		ESMATransactionInformation.ESMATransactionInformationBuilder getOrCreateEsmaTransactionInformation();
		@Override
		ESMATransactionInformation.ESMATransactionInformationBuilder getEsmaTransactionInformation();
		CommonTransactionInformation.CommonTransactionInformationBuilder getOrCreateFcaTransactionInformation();
		@Override
		CommonTransactionInformation.CommonTransactionInformationBuilder getFcaTransactionInformation();
		CommonTransactionInformation.CommonTransactionInformationBuilder getOrCreateMasTransactionInformation();
		@Override
		CommonTransactionInformation.CommonTransactionInformationBuilder getMasTransactionInformation();
		CommonTransactionInformation.CommonTransactionInformationBuilder getOrCreateCsaTransactionInformation();
		@Override
		CommonTransactionInformation.CommonTransactionInformationBuilder getCsaTransactionInformation();
		CommonTransactionInformation.CommonTransactionInformationBuilder getOrCreateHkmaTransactionInformation();
		@Override
		CommonTransactionInformation.CommonTransactionInformationBuilder getHkmaTransactionInformation();
		TransactionInformation.TransactionInformationBuilder setRegimeName(FieldWithMetaRegimeNameEnum regimeName);
		TransactionInformation.TransactionInformationBuilder setRegimeNameValue(RegimeNameEnum regimeName);
		TransactionInformation.TransactionInformationBuilder setSupervisoryBody(FieldWithMetaSupervisoryBodyEnum supervisoryBody);
		TransactionInformation.TransactionInformationBuilder setSupervisoryBodyValue(SupervisoryBodyEnum supervisoryBody);
		TransactionInformation.TransactionInformationBuilder setTransactionIdentifier(TradeIdentifier transactionIdentifier);
		TransactionInformation.TransactionInformationBuilder setPriorTransactionIdentifier(TradeIdentifier priorTransactionIdentifier);
		TransactionInformation.TransactionInformationBuilder setEsmaTransactionInformation(ESMATransactionInformation esmaTransactionInformation);
		TransactionInformation.TransactionInformationBuilder setFcaTransactionInformation(CommonTransactionInformation fcaTransactionInformation);
		TransactionInformation.TransactionInformationBuilder setMasTransactionInformation(CommonTransactionInformation masTransactionInformation);
		TransactionInformation.TransactionInformationBuilder setCsaTransactionInformation(CommonTransactionInformation csaTransactionInformation);
		TransactionInformation.TransactionInformationBuilder setHkmaTransactionInformation(CommonTransactionInformation hkmaTransactionInformation);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processRosetta(path.newSubPath("regimeName"), processor, FieldWithMetaRegimeNameEnum.FieldWithMetaRegimeNameEnumBuilder.class, getRegimeName());
			processRosetta(path.newSubPath("supervisoryBody"), processor, FieldWithMetaSupervisoryBodyEnum.FieldWithMetaSupervisoryBodyEnumBuilder.class, getSupervisoryBody());
			processRosetta(path.newSubPath("transactionIdentifier"), processor, TradeIdentifier.TradeIdentifierBuilder.class, getTransactionIdentifier());
			processRosetta(path.newSubPath("priorTransactionIdentifier"), processor, TradeIdentifier.TradeIdentifierBuilder.class, getPriorTransactionIdentifier());
			processRosetta(path.newSubPath("esmaTransactionInformation"), processor, ESMATransactionInformation.ESMATransactionInformationBuilder.class, getEsmaTransactionInformation());
			processRosetta(path.newSubPath("fcaTransactionInformation"), processor, CommonTransactionInformation.CommonTransactionInformationBuilder.class, getFcaTransactionInformation());
			processRosetta(path.newSubPath("masTransactionInformation"), processor, CommonTransactionInformation.CommonTransactionInformationBuilder.class, getMasTransactionInformation());
			processRosetta(path.newSubPath("csaTransactionInformation"), processor, CommonTransactionInformation.CommonTransactionInformationBuilder.class, getCsaTransactionInformation());
			processRosetta(path.newSubPath("hkmaTransactionInformation"), processor, CommonTransactionInformation.CommonTransactionInformationBuilder.class, getHkmaTransactionInformation());
		}
		

		TransactionInformation.TransactionInformationBuilder prune();
	}

	/*********************** Immutable Implementation of TransactionInformation  ***********************/
	class TransactionInformationImpl implements TransactionInformation {
		private final FieldWithMetaRegimeNameEnum regimeName;
		private final FieldWithMetaSupervisoryBodyEnum supervisoryBody;
		private final TradeIdentifier transactionIdentifier;
		private final TradeIdentifier priorTransactionIdentifier;
		private final ESMATransactionInformation esmaTransactionInformation;
		private final CommonTransactionInformation fcaTransactionInformation;
		private final CommonTransactionInformation masTransactionInformation;
		private final CommonTransactionInformation csaTransactionInformation;
		private final CommonTransactionInformation hkmaTransactionInformation;
		
		protected TransactionInformationImpl(TransactionInformation.TransactionInformationBuilder builder) {
			this.regimeName = ofNullable(builder.getRegimeName()).map(f->f.build()).orElse(null);
			this.supervisoryBody = ofNullable(builder.getSupervisoryBody()).map(f->f.build()).orElse(null);
			this.transactionIdentifier = ofNullable(builder.getTransactionIdentifier()).map(f->f.build()).orElse(null);
			this.priorTransactionIdentifier = ofNullable(builder.getPriorTransactionIdentifier()).map(f->f.build()).orElse(null);
			this.esmaTransactionInformation = ofNullable(builder.getEsmaTransactionInformation()).map(f->f.build()).orElse(null);
			this.fcaTransactionInformation = ofNullable(builder.getFcaTransactionInformation()).map(f->f.build()).orElse(null);
			this.masTransactionInformation = ofNullable(builder.getMasTransactionInformation()).map(f->f.build()).orElse(null);
			this.csaTransactionInformation = ofNullable(builder.getCsaTransactionInformation()).map(f->f.build()).orElse(null);
			this.hkmaTransactionInformation = ofNullable(builder.getHkmaTransactionInformation()).map(f->f.build()).orElse(null);
		}
		
		@Override
		@RosettaAttribute(value="regimeName", isRequired=true)
		@RuneAttribute(value="regimeName", isRequired=true)
		public FieldWithMetaRegimeNameEnum getRegimeName() {
			return regimeName;
		}
		
		@Override
		@RosettaAttribute("supervisoryBody")
		@RuneAttribute("supervisoryBody")
		public FieldWithMetaSupervisoryBodyEnum getSupervisoryBody() {
			return supervisoryBody;
		}
		
		@Override
		@RosettaAttribute("transactionIdentifier")
		@RuneAttribute("transactionIdentifier")
		public TradeIdentifier getTransactionIdentifier() {
			return transactionIdentifier;
		}
		
		@Override
		@RosettaAttribute("priorTransactionIdentifier")
		@RuneAttribute("priorTransactionIdentifier")
		public TradeIdentifier getPriorTransactionIdentifier() {
			return priorTransactionIdentifier;
		}
		
		@Override
		@RosettaAttribute("esmaTransactionInformation")
		@RuneAttribute("esmaTransactionInformation")
		public ESMATransactionInformation getEsmaTransactionInformation() {
			return esmaTransactionInformation;
		}
		
		@Override
		@RosettaAttribute("fcaTransactionInformation")
		@RuneAttribute("fcaTransactionInformation")
		public CommonTransactionInformation getFcaTransactionInformation() {
			return fcaTransactionInformation;
		}
		
		@Override
		@RosettaAttribute("masTransactionInformation")
		@RuneAttribute("masTransactionInformation")
		public CommonTransactionInformation getMasTransactionInformation() {
			return masTransactionInformation;
		}
		
		@Override
		@RosettaAttribute("csaTransactionInformation")
		@RuneAttribute("csaTransactionInformation")
		public CommonTransactionInformation getCsaTransactionInformation() {
			return csaTransactionInformation;
		}
		
		@Override
		@RosettaAttribute("hkmaTransactionInformation")
		@RuneAttribute("hkmaTransactionInformation")
		public CommonTransactionInformation getHkmaTransactionInformation() {
			return hkmaTransactionInformation;
		}
		
		@Override
		public TransactionInformation build() {
			return this;
		}
		
		@Override
		public TransactionInformation.TransactionInformationBuilder toBuilder() {
			TransactionInformation.TransactionInformationBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(TransactionInformation.TransactionInformationBuilder builder) {
			ofNullable(getRegimeName()).ifPresent(builder::setRegimeName);
			ofNullable(getSupervisoryBody()).ifPresent(builder::setSupervisoryBody);
			ofNullable(getTransactionIdentifier()).ifPresent(builder::setTransactionIdentifier);
			ofNullable(getPriorTransactionIdentifier()).ifPresent(builder::setPriorTransactionIdentifier);
			ofNullable(getEsmaTransactionInformation()).ifPresent(builder::setEsmaTransactionInformation);
			ofNullable(getFcaTransactionInformation()).ifPresent(builder::setFcaTransactionInformation);
			ofNullable(getMasTransactionInformation()).ifPresent(builder::setMasTransactionInformation);
			ofNullable(getCsaTransactionInformation()).ifPresent(builder::setCsaTransactionInformation);
			ofNullable(getHkmaTransactionInformation()).ifPresent(builder::setHkmaTransactionInformation);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			TransactionInformation _that = getType().cast(o);
		
			if (!Objects.equals(regimeName, _that.getRegimeName())) return false;
			if (!Objects.equals(supervisoryBody, _that.getSupervisoryBody())) return false;
			if (!Objects.equals(transactionIdentifier, _that.getTransactionIdentifier())) return false;
			if (!Objects.equals(priorTransactionIdentifier, _that.getPriorTransactionIdentifier())) return false;
			if (!Objects.equals(esmaTransactionInformation, _that.getEsmaTransactionInformation())) return false;
			if (!Objects.equals(fcaTransactionInformation, _that.getFcaTransactionInformation())) return false;
			if (!Objects.equals(masTransactionInformation, _that.getMasTransactionInformation())) return false;
			if (!Objects.equals(csaTransactionInformation, _that.getCsaTransactionInformation())) return false;
			if (!Objects.equals(hkmaTransactionInformation, _that.getHkmaTransactionInformation())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (regimeName != null ? regimeName.hashCode() : 0);
			_result = 31 * _result + (supervisoryBody != null ? supervisoryBody.hashCode() : 0);
			_result = 31 * _result + (transactionIdentifier != null ? transactionIdentifier.hashCode() : 0);
			_result = 31 * _result + (priorTransactionIdentifier != null ? priorTransactionIdentifier.hashCode() : 0);
			_result = 31 * _result + (esmaTransactionInformation != null ? esmaTransactionInformation.hashCode() : 0);
			_result = 31 * _result + (fcaTransactionInformation != null ? fcaTransactionInformation.hashCode() : 0);
			_result = 31 * _result + (masTransactionInformation != null ? masTransactionInformation.hashCode() : 0);
			_result = 31 * _result + (csaTransactionInformation != null ? csaTransactionInformation.hashCode() : 0);
			_result = 31 * _result + (hkmaTransactionInformation != null ? hkmaTransactionInformation.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "TransactionInformation {" +
				"regimeName=" + this.regimeName + ", " +
				"supervisoryBody=" + this.supervisoryBody + ", " +
				"transactionIdentifier=" + this.transactionIdentifier + ", " +
				"priorTransactionIdentifier=" + this.priorTransactionIdentifier + ", " +
				"esmaTransactionInformation=" + this.esmaTransactionInformation + ", " +
				"fcaTransactionInformation=" + this.fcaTransactionInformation + ", " +
				"masTransactionInformation=" + this.masTransactionInformation + ", " +
				"csaTransactionInformation=" + this.csaTransactionInformation + ", " +
				"hkmaTransactionInformation=" + this.hkmaTransactionInformation +
			'}';
		}
	}

	/*********************** Builder Implementation of TransactionInformation  ***********************/
	class TransactionInformationBuilderImpl implements TransactionInformation.TransactionInformationBuilder {
	
		protected FieldWithMetaRegimeNameEnum.FieldWithMetaRegimeNameEnumBuilder regimeName;
		protected FieldWithMetaSupervisoryBodyEnum.FieldWithMetaSupervisoryBodyEnumBuilder supervisoryBody;
		protected TradeIdentifier.TradeIdentifierBuilder transactionIdentifier;
		protected TradeIdentifier.TradeIdentifierBuilder priorTransactionIdentifier;
		protected ESMATransactionInformation.ESMATransactionInformationBuilder esmaTransactionInformation;
		protected CommonTransactionInformation.CommonTransactionInformationBuilder fcaTransactionInformation;
		protected CommonTransactionInformation.CommonTransactionInformationBuilder masTransactionInformation;
		protected CommonTransactionInformation.CommonTransactionInformationBuilder csaTransactionInformation;
		protected CommonTransactionInformation.CommonTransactionInformationBuilder hkmaTransactionInformation;
		
		@Override
		@RosettaAttribute(value="regimeName", isRequired=true)
		@RuneAttribute(value="regimeName", isRequired=true)
		public FieldWithMetaRegimeNameEnum.FieldWithMetaRegimeNameEnumBuilder getRegimeName() {
			return regimeName;
		}
		
		@Override
		public FieldWithMetaRegimeNameEnum.FieldWithMetaRegimeNameEnumBuilder getOrCreateRegimeName() {
			FieldWithMetaRegimeNameEnum.FieldWithMetaRegimeNameEnumBuilder result;
			if (regimeName!=null) {
				result = regimeName;
			}
			else {
				result = regimeName = FieldWithMetaRegimeNameEnum.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("supervisoryBody")
		@RuneAttribute("supervisoryBody")
		public FieldWithMetaSupervisoryBodyEnum.FieldWithMetaSupervisoryBodyEnumBuilder getSupervisoryBody() {
			return supervisoryBody;
		}
		
		@Override
		public FieldWithMetaSupervisoryBodyEnum.FieldWithMetaSupervisoryBodyEnumBuilder getOrCreateSupervisoryBody() {
			FieldWithMetaSupervisoryBodyEnum.FieldWithMetaSupervisoryBodyEnumBuilder result;
			if (supervisoryBody!=null) {
				result = supervisoryBody;
			}
			else {
				result = supervisoryBody = FieldWithMetaSupervisoryBodyEnum.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("transactionIdentifier")
		@RuneAttribute("transactionIdentifier")
		public TradeIdentifier.TradeIdentifierBuilder getTransactionIdentifier() {
			return transactionIdentifier;
		}
		
		@Override
		public TradeIdentifier.TradeIdentifierBuilder getOrCreateTransactionIdentifier() {
			TradeIdentifier.TradeIdentifierBuilder result;
			if (transactionIdentifier!=null) {
				result = transactionIdentifier;
			}
			else {
				result = transactionIdentifier = TradeIdentifier.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("priorTransactionIdentifier")
		@RuneAttribute("priorTransactionIdentifier")
		public TradeIdentifier.TradeIdentifierBuilder getPriorTransactionIdentifier() {
			return priorTransactionIdentifier;
		}
		
		@Override
		public TradeIdentifier.TradeIdentifierBuilder getOrCreatePriorTransactionIdentifier() {
			TradeIdentifier.TradeIdentifierBuilder result;
			if (priorTransactionIdentifier!=null) {
				result = priorTransactionIdentifier;
			}
			else {
				result = priorTransactionIdentifier = TradeIdentifier.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("esmaTransactionInformation")
		@RuneAttribute("esmaTransactionInformation")
		public ESMATransactionInformation.ESMATransactionInformationBuilder getEsmaTransactionInformation() {
			return esmaTransactionInformation;
		}
		
		@Override
		public ESMATransactionInformation.ESMATransactionInformationBuilder getOrCreateEsmaTransactionInformation() {
			ESMATransactionInformation.ESMATransactionInformationBuilder result;
			if (esmaTransactionInformation!=null) {
				result = esmaTransactionInformation;
			}
			else {
				result = esmaTransactionInformation = ESMATransactionInformation.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("fcaTransactionInformation")
		@RuneAttribute("fcaTransactionInformation")
		public CommonTransactionInformation.CommonTransactionInformationBuilder getFcaTransactionInformation() {
			return fcaTransactionInformation;
		}
		
		@Override
		public CommonTransactionInformation.CommonTransactionInformationBuilder getOrCreateFcaTransactionInformation() {
			CommonTransactionInformation.CommonTransactionInformationBuilder result;
			if (fcaTransactionInformation!=null) {
				result = fcaTransactionInformation;
			}
			else {
				result = fcaTransactionInformation = CommonTransactionInformation.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("masTransactionInformation")
		@RuneAttribute("masTransactionInformation")
		public CommonTransactionInformation.CommonTransactionInformationBuilder getMasTransactionInformation() {
			return masTransactionInformation;
		}
		
		@Override
		public CommonTransactionInformation.CommonTransactionInformationBuilder getOrCreateMasTransactionInformation() {
			CommonTransactionInformation.CommonTransactionInformationBuilder result;
			if (masTransactionInformation!=null) {
				result = masTransactionInformation;
			}
			else {
				result = masTransactionInformation = CommonTransactionInformation.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("csaTransactionInformation")
		@RuneAttribute("csaTransactionInformation")
		public CommonTransactionInformation.CommonTransactionInformationBuilder getCsaTransactionInformation() {
			return csaTransactionInformation;
		}
		
		@Override
		public CommonTransactionInformation.CommonTransactionInformationBuilder getOrCreateCsaTransactionInformation() {
			CommonTransactionInformation.CommonTransactionInformationBuilder result;
			if (csaTransactionInformation!=null) {
				result = csaTransactionInformation;
			}
			else {
				result = csaTransactionInformation = CommonTransactionInformation.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("hkmaTransactionInformation")
		@RuneAttribute("hkmaTransactionInformation")
		public CommonTransactionInformation.CommonTransactionInformationBuilder getHkmaTransactionInformation() {
			return hkmaTransactionInformation;
		}
		
		@Override
		public CommonTransactionInformation.CommonTransactionInformationBuilder getOrCreateHkmaTransactionInformation() {
			CommonTransactionInformation.CommonTransactionInformationBuilder result;
			if (hkmaTransactionInformation!=null) {
				result = hkmaTransactionInformation;
			}
			else {
				result = hkmaTransactionInformation = CommonTransactionInformation.builder();
			}
			
			return result;
		}
		
		@RosettaAttribute(value="regimeName", isRequired=true)
		@RuneAttribute(value="regimeName", isRequired=true)
		@Override
		public TransactionInformation.TransactionInformationBuilder setRegimeName(FieldWithMetaRegimeNameEnum _regimeName) {
			this.regimeName = _regimeName == null ? null : _regimeName.toBuilder();
			return this;
		}
		
		@Override
		public TransactionInformation.TransactionInformationBuilder setRegimeNameValue(RegimeNameEnum _regimeName) {
			this.getOrCreateRegimeName().setValue(_regimeName);
			return this;
		}
		
		@RosettaAttribute("supervisoryBody")
		@RuneAttribute("supervisoryBody")
		@Override
		public TransactionInformation.TransactionInformationBuilder setSupervisoryBody(FieldWithMetaSupervisoryBodyEnum _supervisoryBody) {
			this.supervisoryBody = _supervisoryBody == null ? null : _supervisoryBody.toBuilder();
			return this;
		}
		
		@Override
		public TransactionInformation.TransactionInformationBuilder setSupervisoryBodyValue(SupervisoryBodyEnum _supervisoryBody) {
			this.getOrCreateSupervisoryBody().setValue(_supervisoryBody);
			return this;
		}
		
		@RosettaAttribute("transactionIdentifier")
		@RuneAttribute("transactionIdentifier")
		@Override
		public TransactionInformation.TransactionInformationBuilder setTransactionIdentifier(TradeIdentifier _transactionIdentifier) {
			this.transactionIdentifier = _transactionIdentifier == null ? null : _transactionIdentifier.toBuilder();
			return this;
		}
		
		@RosettaAttribute("priorTransactionIdentifier")
		@RuneAttribute("priorTransactionIdentifier")
		@Override
		public TransactionInformation.TransactionInformationBuilder setPriorTransactionIdentifier(TradeIdentifier _priorTransactionIdentifier) {
			this.priorTransactionIdentifier = _priorTransactionIdentifier == null ? null : _priorTransactionIdentifier.toBuilder();
			return this;
		}
		
		@RosettaAttribute("esmaTransactionInformation")
		@RuneAttribute("esmaTransactionInformation")
		@Override
		public TransactionInformation.TransactionInformationBuilder setEsmaTransactionInformation(ESMATransactionInformation _esmaTransactionInformation) {
			this.esmaTransactionInformation = _esmaTransactionInformation == null ? null : _esmaTransactionInformation.toBuilder();
			return this;
		}
		
		@RosettaAttribute("fcaTransactionInformation")
		@RuneAttribute("fcaTransactionInformation")
		@Override
		public TransactionInformation.TransactionInformationBuilder setFcaTransactionInformation(CommonTransactionInformation _fcaTransactionInformation) {
			this.fcaTransactionInformation = _fcaTransactionInformation == null ? null : _fcaTransactionInformation.toBuilder();
			return this;
		}
		
		@RosettaAttribute("masTransactionInformation")
		@RuneAttribute("masTransactionInformation")
		@Override
		public TransactionInformation.TransactionInformationBuilder setMasTransactionInformation(CommonTransactionInformation _masTransactionInformation) {
			this.masTransactionInformation = _masTransactionInformation == null ? null : _masTransactionInformation.toBuilder();
			return this;
		}
		
		@RosettaAttribute("csaTransactionInformation")
		@RuneAttribute("csaTransactionInformation")
		@Override
		public TransactionInformation.TransactionInformationBuilder setCsaTransactionInformation(CommonTransactionInformation _csaTransactionInformation) {
			this.csaTransactionInformation = _csaTransactionInformation == null ? null : _csaTransactionInformation.toBuilder();
			return this;
		}
		
		@RosettaAttribute("hkmaTransactionInformation")
		@RuneAttribute("hkmaTransactionInformation")
		@Override
		public TransactionInformation.TransactionInformationBuilder setHkmaTransactionInformation(CommonTransactionInformation _hkmaTransactionInformation) {
			this.hkmaTransactionInformation = _hkmaTransactionInformation == null ? null : _hkmaTransactionInformation.toBuilder();
			return this;
		}
		
		@Override
		public TransactionInformation build() {
			return new TransactionInformation.TransactionInformationImpl(this);
		}
		
		@Override
		public TransactionInformation.TransactionInformationBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public TransactionInformation.TransactionInformationBuilder prune() {
			if (regimeName!=null && !regimeName.prune().hasData()) regimeName = null;
			if (supervisoryBody!=null && !supervisoryBody.prune().hasData()) supervisoryBody = null;
			if (transactionIdentifier!=null && !transactionIdentifier.prune().hasData()) transactionIdentifier = null;
			if (priorTransactionIdentifier!=null && !priorTransactionIdentifier.prune().hasData()) priorTransactionIdentifier = null;
			if (esmaTransactionInformation!=null && !esmaTransactionInformation.prune().hasData()) esmaTransactionInformation = null;
			if (fcaTransactionInformation!=null && !fcaTransactionInformation.prune().hasData()) fcaTransactionInformation = null;
			if (masTransactionInformation!=null && !masTransactionInformation.prune().hasData()) masTransactionInformation = null;
			if (csaTransactionInformation!=null && !csaTransactionInformation.prune().hasData()) csaTransactionInformation = null;
			if (hkmaTransactionInformation!=null && !hkmaTransactionInformation.prune().hasData()) hkmaTransactionInformation = null;
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (getRegimeName()!=null) return true;
			if (getSupervisoryBody()!=null) return true;
			if (getTransactionIdentifier()!=null && getTransactionIdentifier().hasData()) return true;
			if (getPriorTransactionIdentifier()!=null && getPriorTransactionIdentifier().hasData()) return true;
			if (getEsmaTransactionInformation()!=null && getEsmaTransactionInformation().hasData()) return true;
			if (getFcaTransactionInformation()!=null && getFcaTransactionInformation().hasData()) return true;
			if (getMasTransactionInformation()!=null && getMasTransactionInformation().hasData()) return true;
			if (getCsaTransactionInformation()!=null && getCsaTransactionInformation().hasData()) return true;
			if (getHkmaTransactionInformation()!=null && getHkmaTransactionInformation().hasData()) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public TransactionInformation.TransactionInformationBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			TransactionInformation.TransactionInformationBuilder o = (TransactionInformation.TransactionInformationBuilder) other;
			
			merger.mergeRosetta(getRegimeName(), o.getRegimeName(), this::setRegimeName);
			merger.mergeRosetta(getSupervisoryBody(), o.getSupervisoryBody(), this::setSupervisoryBody);
			merger.mergeRosetta(getTransactionIdentifier(), o.getTransactionIdentifier(), this::setTransactionIdentifier);
			merger.mergeRosetta(getPriorTransactionIdentifier(), o.getPriorTransactionIdentifier(), this::setPriorTransactionIdentifier);
			merger.mergeRosetta(getEsmaTransactionInformation(), o.getEsmaTransactionInformation(), this::setEsmaTransactionInformation);
			merger.mergeRosetta(getFcaTransactionInformation(), o.getFcaTransactionInformation(), this::setFcaTransactionInformation);
			merger.mergeRosetta(getMasTransactionInformation(), o.getMasTransactionInformation(), this::setMasTransactionInformation);
			merger.mergeRosetta(getCsaTransactionInformation(), o.getCsaTransactionInformation(), this::setCsaTransactionInformation);
			merger.mergeRosetta(getHkmaTransactionInformation(), o.getHkmaTransactionInformation(), this::setHkmaTransactionInformation);
			
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			TransactionInformation _that = getType().cast(o);
		
			if (!Objects.equals(regimeName, _that.getRegimeName())) return false;
			if (!Objects.equals(supervisoryBody, _that.getSupervisoryBody())) return false;
			if (!Objects.equals(transactionIdentifier, _that.getTransactionIdentifier())) return false;
			if (!Objects.equals(priorTransactionIdentifier, _that.getPriorTransactionIdentifier())) return false;
			if (!Objects.equals(esmaTransactionInformation, _that.getEsmaTransactionInformation())) return false;
			if (!Objects.equals(fcaTransactionInformation, _that.getFcaTransactionInformation())) return false;
			if (!Objects.equals(masTransactionInformation, _that.getMasTransactionInformation())) return false;
			if (!Objects.equals(csaTransactionInformation, _that.getCsaTransactionInformation())) return false;
			if (!Objects.equals(hkmaTransactionInformation, _that.getHkmaTransactionInformation())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (regimeName != null ? regimeName.hashCode() : 0);
			_result = 31 * _result + (supervisoryBody != null ? supervisoryBody.hashCode() : 0);
			_result = 31 * _result + (transactionIdentifier != null ? transactionIdentifier.hashCode() : 0);
			_result = 31 * _result + (priorTransactionIdentifier != null ? priorTransactionIdentifier.hashCode() : 0);
			_result = 31 * _result + (esmaTransactionInformation != null ? esmaTransactionInformation.hashCode() : 0);
			_result = 31 * _result + (fcaTransactionInformation != null ? fcaTransactionInformation.hashCode() : 0);
			_result = 31 * _result + (masTransactionInformation != null ? masTransactionInformation.hashCode() : 0);
			_result = 31 * _result + (csaTransactionInformation != null ? csaTransactionInformation.hashCode() : 0);
			_result = 31 * _result + (hkmaTransactionInformation != null ? hkmaTransactionInformation.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "TransactionInformationBuilder {" +
				"regimeName=" + this.regimeName + ", " +
				"supervisoryBody=" + this.supervisoryBody + ", " +
				"transactionIdentifier=" + this.transactionIdentifier + ", " +
				"priorTransactionIdentifier=" + this.priorTransactionIdentifier + ", " +
				"esmaTransactionInformation=" + this.esmaTransactionInformation + ", " +
				"fcaTransactionInformation=" + this.fcaTransactionInformation + ", " +
				"masTransactionInformation=" + this.masTransactionInformation + ", " +
				"csaTransactionInformation=" + this.csaTransactionInformation + ", " +
				"hkmaTransactionInformation=" + this.hkmaTransactionInformation +
			'}';
		}
	}
}
