package drr.regulation.cftc.rewrite.margin;

import cdm.base.staticdata.asset.common.ISOCurrencyCodeEnum;
import cdm.base.staticdata.party.PartyIdentifierTypeEnum;
import com.google.common.collect.ImmutableList;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import com.rosetta.util.ListEquals;
import drr.regulation.cftc.rewrite.margin.meta.CFTCMarginReportMeta;
import drr.regulation.common.SupervisoryBodyEnum;
import drr.standards.iso.CollateralisationType3Code__1;
import drr.standards.iso.MarginActionEnum;
import java.math.BigDecimal;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import static java.util.Optional.ofNullable;

/**
 * @version 7.0.0-dev.73
 */
@RosettaDataType(value="CFTCMarginReport", builder=CFTCMarginReport.CFTCMarginReportBuilderImpl.class, version="7.0.0-dev.73")
@RuneDataType(value="CFTCMarginReport", model="drr", builder=CFTCMarginReport.CFTCMarginReportBuilderImpl.class, version="7.0.0-dev.73")
public interface CFTCMarginReport extends RosettaModelObject {

	CFTCMarginReportMeta metaData = new CFTCMarginReportMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "13" * field "Counterparty 1 (reporting counterparty)"
	 *
	 * Provision Identifier of the counterparty to an OTC derivative transaction who is fulfilling its reporting obligation via the report in question. In jurisdictions where both parties must report the transaction, the identifier of Counterparty 1 always identifies the reporting counterparty. In the case of an allocated derivative transaction executed by a fund manager on behalf of a fund, the fund, and not the fund manager is reported as the counterparty.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "13" * field "Counterparty 1" * footnote "24"
	 *
	 * Provision Throughout this Technical Specification, references to 'OTC derivatives' in CDE data elements should be read to mean swaps.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "13" * field "Counterparty 1" * footnote "25"
	 *
	 * Provision  ROC Statement  Individuals Acting in a Business Capacity, ROC Statement - Individuals Acting in a Business Capacity.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup EUUKNAPeerReviewGroup ISDA EU/UK/North America Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended EU/UK EMIR and North America transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20250320" * field "Counterparty 1 (reporting counterparty)"
	 *
	 * Provision The same CDE Counterparty 1 extraction is required.
	 *
	 */
	String getCounterparty1();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "14" * field "Counterparty 2"
	 *
	 * Provision Identifier of the second counterparty to an OTC derivative transaction. In the case of an allocated derivative transaction executed by a fund manager on behalf of a fund, the fund, and not the fund manager is reported as the counterparty.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "14" * field "Counterparty 2" * footnote "26"
	 *
	 * Provision In the case of a swap transaction for pre-allocated block executed by a fund manager on behalf of a fund, the fund manager is reported as the counterparty.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "14" * field "Counterparty 2" * footnote "27"
	 *
	 * Provision Only one counterparty should be reported. In cases where multiple counterparties are legally responsible as the second counterparty (such as an obligated group, for example), report only one of the counterparties and use the same counterparty for all continuation data and lifecycle events.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "14" * field "Counterparty 2" * footnote "28"
	 *
	 * Provision ROC Statement  Individuals Acting in a Business Capacity, ROC Statement - Individuals Acting in a Business Capacity
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "14" * field "Counterparty 2" * footnote "29"
	 *
	 * Provision Throughout this Technical Specification, Privacy Law Identifier (PLI) has been added as an allowable value for all Counterparty 2 (#14) associated data elements.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup EUUKNAPeerReviewGroup ISDA EU/UK/North America Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended EU/UK EMIR and North America transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20250320" * field "Counterparty 2 (non-reporting counterparty)"
	 *
	 * Provision The same CDE Counterparty 2 extraction is required.
	 *
	 */
	String getCounterparty2();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "15" * field "Counterparty 2 Identifier Source"
	 *
	 * Provision Source used to identify the Counterparty 2.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "15" * field "Counterparty 2 Identifier Source" * footnote "30"
	 *
	 * Provision Throughout this Technical Specification, for references to 'Privacy Law Identifiers', refer to DMO Letter No. 17-16, http://www.cftc.gov/idc/groups/public/@lrlettergeneral/documents/letter/17-16.pdf
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup EUUKNAPeerReviewGroup ISDA EU/UK/North America Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended EU/UK EMIR and North America transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20250320" * field "Counterparty 2 Identifier Source"
	 *
	 * Provision Improvement on the logic was done to consider first if person exists. If not, then consider party.
	 *
	 */
	String getCounterparty2IdentifierSource();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "22" * field "Submitter Identifier"
	 *
	 * Provision Identifier of the entity submitting the data to the swap data repository (SDR). The Submitter identifier will be the same as the reporting counterparty or swap execution facility (SEF), unless they use a third-party service provider to submit the data to SDR in which case, report the identifier of the third-party service provider.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220309"
	 *
	 * Provision Check whether trade is executed in a SEF first. If it is, SEF is obliged to be the Submitter. Then check for Reporting Party and Data Submitter.
	 *
	 */
	String getSubmitterIdentifier();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "26" * field "Action Type"
	 *
	 * Provision Type of action taken on the swap transaction or type of end-of-day reporting. Actions may include, but are not limited to, new, modify, correct, error, terminate, revive, transfer out, valuation, and collateral.
	                 - New: An action that reports a new swap transaction. It applies to the first message relating to a new USI or UTI.
	                 - Modify: An action that modifies the state of a previously submitted transaction (e.g., credit event) or changes a term of a previously submitted transaction due to a newly negotiated modification (amendment) or updates previously missing information (e.g., post price swap). It does not include correction of a previous transaction.
	                 - Correct: An action that corrects erroneous data of a previously submitted transaction.
	                 - Error: An action of cancellation of a wrongly submitted entire transaction in case it never came into existence or was not subject to part 43/part 45 reporting requirements but was reported erroneously.
	                 - Terminate: An action that closes an existing transaction because of a new event (e.g., Compression, Novation). This does not apply to transactions that terminate at contractual maturity date.
	                 - Revive: An action that reinstates a swap transaction that was reported as error or terminated by mistake.
	                 - Transfer out: An action that transfers swap transaction from one SDR to another SDR (change of swap data repository).
	                 - Valuation: An update to valuation data. There will be no corresponding Event type.
	                 - Collateral: An update to collateral margin data. There will be no corresponding Event type. Refer to appendix F for event model sample scenarios
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "26" * field "Action Type" * footnote "33"
	 *
	 * Provision Only one Action type value is allowed per submission. Multiple Action type values should not be submitted in one transaction report. For example, if a data element needs to be corrected on a previously submitted transaction that is getting terminated, the Correct (CORR) value should be submitted as a separate submission prior to the submission of the Terminate (TERM) transaction.
	 *
	 */
	MarginActionEnum getActionType();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "30" * field "Event Timestamp"
	 *
	 * Provision Date and time of occurrence of the event as determined by the reporting counterparty or a service provider. In the case of a clearing event, date and time when the original swap is accepted by the derivative clearing organization (DCO) for clearing and recorded by the DCOs system should be reported in this data element. The time element is as specific as technologically practicable.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "30" * field "Event Timestamp" * footnote "38"
	 *
	 * Provision In the case of collateral update (Action type = 'MARU'), this is the date for which the information contained in the report is provided. Time portion should be reported as '00:00:00'.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "30" * field "Event Timestamp" * footnote "39"
	 *
	 * Provision Both the date and time portion are required to be reported. If the time portion is not available, report '00:00:00' for the time portion.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20240117"
	 *
	 * Provision DRR users are expected to use eventCreationDateTime as qualifier to the timestamp they have to report as the Event Timestamp
	 *
	 */
	ZonedDateTime getEventTimestamp();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "97" * field "Reporting Timestamp"
	 *
	 * Provision Date and time of the submission of the report to the trade repository
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "97" * field "Reporting Timestamp" * footnote "74"
	 *
	 * Provision Reporting timestamp (#97) is recorded and reported by the submitter.
	 *
	 */
	ZonedDateTime getReportingTimestamp();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "102" * field "Unique Swap Identifier (USI)"
	 *
	 * Provision The USI is a unique identifier assigned to all swap transactions which identifies the transaction (the swap and its counterparties) uniquely throughout its duration. It consists of a namespace and a transaction identifier.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "102" * field "Unique Swap Identifier (USI)" * footnote "76"
	 *
	 * Provision Throughout this Technical Specification, references to 'CFTC USI Data Standard' should refer to the USI Data Standard, https://www.cftc.gov/sites/default/files/idc/groups/public/@swaps/documents/dfsubmission/usidatastandards100112.pdf
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220309"
	 *
	 * Provision Rule does not model UTI > USI priority defined in CFTC specification.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220309"
	 *
	 * Provision Firms do expect to report USI/UTI in both formats: as a single identifier field as well as using Issuer + identifier. The functional model needs to handle both scenarios.
	 *
	 */
	String getUniqueSwapIdentifier();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "103" * field "Unique Transaction Identifier (UTI)"
	 *
	 * Provision A unique identifier assigned to all swap transactions which identifies the swap uniquely throughout its lifecycle and used for all recordkeeping and all swap data reporting pursuant to para 45.5. A UTI is comprised of the LEI of the generating entity and a unique alphanumeric code.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220309"
	 *
	 * Provision Rule does not model UTI > USI priority defined in CFTC specification.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220309"
	 *
	 * Provision Firms do expect to report USI/UTI in both formats: as a single identifier field as well as using Issuer + identifier. The functional model needs to handle both scenarios.
	 *
	 */
	String getUniqueTransactionIdentifier();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "114" * field "Affiliated Counterparty For Margin And Capital Indicator"
	 *
	 * Provision Indicator of whether the current counterparty is deemed an affiliate for U.S. margin and capital rules (as per para 23.159).
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup USComplianceGroup US Compliance Working Group "This is an ISDA US Working Group." 
	 * date "20220802"
	 *
	 * Provision Not returning anything until it gets modelled
	 *
	 */
	Boolean getAffiliatedCounterpartyForMarginAndCapitalIndicator();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "115" * field "Collateralisation Category"
	 *
	 * Provision Indicator of whether a collateral agreement (or collateral agreements) between the counterparties exists (uncollateralised/partially collateralised/one-way collateralised/fully collateralised). This data element is provided for each transaction or each portfolio, depending on whether the collateralisation is performed at the transaction or portfolio level, and is applicable to both cleared and uncleared transactions.
	 *
	 */
	CollateralisationType3Code__1 getCollateralisationCategory();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "116" * field "Initial Margin Collateral Portfolio Code"
	 *
	 * Provision If collateral is reported on a portfolio basis, a unique code assigned by the reporting counterparty to the portfolio that tracks the aggregate initial margin of a set of open swap transactions. This data element is not applicable if the collateralisation was performed on a transaction level basis, or if there is no collateral agreement, or if no collateral is posted or received. The portfolio code is required for both collateral reporting and valuation reporting in order to link the 2 data sets.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "116" * field "Initial Margin Collateral Portfolio Code" * footnote "82"
	 *
	 * Provision If collateralization was performed on a transaction level basis, 'TRANSACTION-LEVEL' is accepted.  “NOTAPPLICABLE” is accepted if (i) collateralization was performed on a portfolio basis and there is no IM portfolio code, or (ii) it is a submission from a DCO. 
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "116" * field "Initial Margin Collateral Portfolio Code" * footnote "83"
	 *
	 * Provision  The M in the Part 43/45 Asset Class column is for collateral and valuation reporting only.
	 *
	 */
	String getInitialMarginCollateralPortfolioCode();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "117" * field "Portfolio Containing Non Reportable Component Indicator"
	 *
	 * Provision If collateral is reported on a portfolio basis, indicator of whether the collateral portfolio includes swap transactions exempt from reporting.
	 *
	 */
	Boolean getPortfolioContainingNonReportableComponentIndicator();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "118" * field "Initial Margin Posted By The Reporting Counterparty Post Haircut"
	 *
	 * Provision Monetary value of initial margin that has been posted by the reporting counterparty, including any margin that is in transit and pending settlement unless inclusion of such margin is not allowed under the jurisdictional requirements. If the collateralisation is performed at portfolio level, the initial margin posted relates to the whole portfolio; if the collateralisation is performed for single transactions, the initial margin posted relates to such single transaction. This refers to the total current value of the initial margin after application of the haircut (if applicable), rather than to its daily change. The data element refers both to uncleared and centrally cleared transactions. For centrally cleared transactions, the data element does not include default fund contributions, nor collateral posted against liquidity provisions to the central counterparty, i.e., committed credit lines. If the initial margin posted is denominated in more than one currency, those amounts are converted into a single currency chosen by the reporting counterparty and reported as one total value.
	 *
	 */
	BigDecimal getInitialMarginPostedByTheReportingCounterpartyPostHaircut();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "119" * field "Initial Margin Posted By The Reporting Counterparty (Pre-Haircut)"
	 *
	 * Provision Monetary value of initial margin that has been posted by the reporting counterparty, including any margin that is in transit and pending settlement unless inclusion of such margin is not allowed under the jurisdictional requirements. If the collateralisation is performed at portfolio level, the initial margin posted relates to the whole portfolio; if the collateralisation is performed for single transactions, the initial margin posted relates to such single transaction. This refers to the total current value of the initial margin, rather than to its daily change. The data element refers both to uncleared and centrally cleared transactions. For centrally cleared transactions, the data element does not include default fund contributions, nor collateral posted against liquidity provisions to the central counterparty, i.e., committed credit lines. If the initial margin posted is denominated in more than one currency, those amounts are converted into a single currency chosen by the reporting counterparty and reported as one total value.
	 *
	 */
	BigDecimal getInitialMarginPostedByTheReportingCounterpartyPreHaircut();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "120" * field "Currency Of Initial Margin Posted"
	 *
	 * Provision Currency in which the initial margin posted is denominated. If the initial margin posted is denominated in more than one currency, this data element reflects one of those currencies into which the reporting counterparty has chosen to convert all the values of posted initial margins.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "120" * field "Currency Of Initial Margin Posted" * footnote "79"
	 *
	 * Provision For portfolio with multiple currencies, it must be converted in to a single currency chosen by the reporting counterparty and reported.
	 *
	 */
	ISOCurrencyCodeEnum getCurrencyOfInitialMarginPosted();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "121" * field "Initial Margin Collected By The Reporting Counterparty (Post-Haircut)"
	 *
	 * Provision Monetary value of initial margin that has been collected by the reporting counterparty, including any margin that is in transit and pending settlement unless inclusion of such margin is not allowed under the jurisdictional requirements. If the collateralisation is performed at portfolio level, the initial margin collected relates to the whole portfolio; if the collateralisation is performed for single transactions, the initial margin collected relates to such single transaction. This refers to the total current value of the initial margin after application of the haircut (if applicable), rather than to its daily change. The data element refers both to uncleared and centrally cleared transactions. For centrally cleared transactions, the data element does not include collateral collected by the central counterparty as part of its investment activity. If the initial margin collected is denominated in more than one currency, those amounts are converted into a single currency chosen by the reporting counterparty and reported as one total value.
	 *
	 */
	BigDecimal getInitialMarginCollectedByTheReportingCounterpartyPostHaircut();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "122" * field "Initial Margin Collected By The Reporting Counterparty (Pre-Haircut)"
	 *
	 * Provision Monetary value of initial margin that has been collected by the reporting counterparty, including any margin that is in transit and pending settlement unless inclusion of such margin is not allowed under the jurisdictional requirements. If the collateralisation is performed at portfolio level, the initial margin collected relates to the whole portfolio; if the collateralisation is performed for single transactions, the initial margin collected relates to such single transaction. This refers to the total current value of the initial margin, rather than to its daily change. The data element refers both to uncleared and centrally cleared transactions. For centrally cleared transactions, the data element does not include collateral collected by the central counterparty as part of its investment activity. If the initial margin collected is denominated in more than one currency, those amounts are converted into a single currency chosen by the reporting counterparty and reported as one total value.
	 *
	 */
	BigDecimal getInitialMarginCollectedByTheReportingCounterpartyPreHaircut();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "123" * field "Currency Of Initial Margin Collected"
	 *
	 * Provision Currency in which the initial margin collected is denominated. If the initial margin collected is denominated in more than one currency, this data element reflects one of those currencies into which the reporting counterparty has chosen to convert all the values of collected initial margins.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "123" * field "Currency Of Initial Margin Collected" * footnote "84"
	 *
	 * Provision For portfolio with multiple currencies, it must be converted in to a single currency chosen by the reporting counterparty and reported.
	 *
	 */
	ISOCurrencyCodeEnum getCurrencyOfInitialMarginCollected();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "124" * field "Variation Margin Collateral Portfolio Code"
	 *
	 * Provision If collateral is reported on a portfolio basis, a unique code assigned by the reporting counterparty to the portfolio that tracks the aggregate variation margin related to a set of open swap transactions. This data element is not applicable if the collateralisation was performed on a transaction level basis, or if there is no collateral agreement, or if no collateral is posted or received. The portfolio code is required for both collateral reporting and valuation reporting in order to link the 2 data sets.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "124" * field "Variation Margin Collateral Portfolio Code" * footnote "90"
	 *
	 * Provision If collateralization was performed on a transaction level basis, 'TRANSACTION-LEVEL' is accepted. “NOTAPPLICABLE” is accepted if (i) collateralization was performed on a portfolio basis and there is no VM portfolio code, or (ii) it is a submission from a DCO.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "124" * field "Variation Margin Collateral Portfolio Code" * footnote "91"
	 *
	 * Provision The 'M' in the 'Part 43/45 Asset Class' column is for collateral and valuation reporting only.
	 *
	 */
	String getVariationMarginCollateralPortfolioCode();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "125" * field "Variation Margin Posted By The Reporting Counterparty (Pre-Haircut)"
	 *
	 * Provision Monetary value of the variation margin posted by the reporting counterparty (including the cash-settled one), and including any margin that is in transit and pending settlement unless inclusion of such margin is not allowed under the jurisdictional requirements. Contingent variation margin is not included. If the collateralisation is performed at portfolio level, the variation margin posted relates to the whole portfolio; if the collateralisation is performed for single transactions, the variation margin posted relates to such single transaction. This data element refers to the total current value of the variation margin, cumulated since the first reporting of variation margins posted for the portfolio/transaction. If the variation margin posted is denominated in more than one currency, those amounts are converted into a single currency chosen by the reporting counterparty and reported as one total value.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "125" * field "Variation Margin Posted By The Reporting Counterparty (Pre-Haircut)" * footnote "87"
	 *
	 * Provision This data element must be reported daily regardless of whether there is a change in the value since the last reporting.
	 *
	 */
	BigDecimal getVariationMarginPostedByTheReportingCounterpartyPreHaircut();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "126" * field "Currency Of Variation Margin Posted"
	 *
	 * Provision Currency in which the variation margin posted is denominated. If the variation margin posted is denominated in more than one currency, this data element reflects one of those currencies into which the reporting counterparty has chosen to convert all the values of posted variation margins.
	 *
	 */
	ISOCurrencyCodeEnum getCurrencyOfVariationMarginPosted();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "127" * field "Variation Margin Collected By The Reporting Counterparty (Pre-Haircut)"
	 *
	 * Provision Monetary value of the variation margin collected by the reporting counterparty (including the cash-settled one), and including any margin that is in transit and pending settlement unless inclusion of such margin is not allowed under the jurisdictional requirements. Contingent variation margin is not included. If the collateralisation is performed at portfolio level, the variation margin collected relates to the whole portfolio; if the collateralisation is performed for single transactions, the variation margin collected relates to such single transaction. This refers to the total current value of the variation margin, cumulated since the first reporting of collected variation margins for the portfolio/ transaction. If the variation margin collected is denominated in more than one currency, those amounts are converted into a single currency chosen by the reporting counterparty and reported as one total value.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "127" * field "Variation Margin Collected By The Reporting Counterparty (Pre-Haircut)" * footnote "88"
	 *
	 * Provision This data element must be reported daily regardless of whether there is a change in the value since the last reporting.
	 *
	 */
	BigDecimal getVariationMarginCollectedByTheReportingCounterpartyPreHaircut();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin   
	 * appendix "1" * dataElement "128" * field "Currency Of Variation Margin Collected"
	 *
	 * Provision Currency in which the variation margin collected is denominated. If the variation margin collected is denominated in more than one currency, this data element reflects one of those currencies into which the reporting counterparty has chosen to convert all the values of collected variation margins.
	 *
	 */
	ISOCurrencyCodeEnum getCurrencyOfVariationMarginCollected();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Trade Party 1 - ID Type"
	 *
	 * Provision GTR required Control Field.
	 *
	 */
	PartyIdentifierTypeEnum getTradeParty1IDType();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Submitting Party - ID Type"
	 *
	 * Provision GTR required Control Field.
	 *
	 */
	PartyIdentifierTypeEnum getSubmittingPartyIDType();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "USI ID Prefix"
	 *
	 * Provision The USI is a unique identifier assigned to all swap transactions which identifies the transaction (the swap and its counterparties) uniquely throughout its duration. It consists of a namespace and a transaction identifier.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup EUUKNAPeerReviewGroup ISDA EU/UK/North America Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended EU/UK EMIR and North America transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20250327" * field "USI ID Prefix"
	 *
	 * Provision The group agreed on not considering USI ID Prefix due to it comes from an old format and it had been agreed to only consider one field, that is USI ID, considering the concatenation of both fields. Thus, USI ID is the only field that needs to be populated.
	 *
	 */
	String getUsiIDPrefix();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "UTI ID Prefix"
	 *
	 * Provision A unique identifier assigned to all swap transactions which identifies the swap uniquely throughout its lifecycle and used for all recordkeeping and all swap data reporting pursuant to §45.5. A UTI is comprised of the LEI of the generating entity and a unique alphanumeric code.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup EUUKNAPeerReviewGroup ISDA EU/UK/North America Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended EU/UK EMIR and North America transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20250327" * field "UTI ID Prefix"
	 *
	 * Provision The group agreed on not considering UTI ID Prefix due to it comes from an old format and it had been agreed to only consider one field, that is UTI ID, considering the concatenation of both fields. Thus, UTI ID is the only field that needs to be populated.
	 *
	 */
	String getUtiIDPrefix();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Trade Party 1 - Reporting Destination"
	 *
	 * Provision Needed to report Jurisdiction.
	 *
	 */
	List<SupervisoryBodyEnum> getTradeParty1ReportingDestination();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Comment 1"
	 *
	 * Provision For CSV submissions if  populated with an * the entire line is treated as comment.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup EUUKNAPeerReviewGroup ISDA EU/UK/North America Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended EU/UK EMIR and North America transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20250507" * field "Comment 1"
	 *
	 * Provision Empty because CSV submission is still not supported by current DRR model. This rule will be revisited once Harmonized CSV projection is included.
	 *
	 */
	String getComment1();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Message ID"
	 *
	 * Provision Purpose of this field is to allow firms to submit a unique ID that will allow them to tie in specific submissions/modifications. For example if 5 modifications are submitted on a specific PET, this will allow the firm to ensure they receive responses for all 5 modifications.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup EUUKNAPeerReviewGroup ISDA EU/UK/North America Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended EU/UK EMIR and North America transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20250604" * field "Message ID"
	 *
	 * Provision Each Message ID must be unique, irrespective of the report type. This unique identification is achieved by concatenating with the Reporting Timestamp field.
	 *
	 */
	String getMessageID();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Message Type"
	 *
	 * Provision Need to indicate what message is being submitted to GTR.
	 *
	 */
	String getMessageType();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Margin    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Version"
	 *
	 * Provision Memo field. Used to identify the asset class and template type of submission.
	 *
	 */
	String getVersion();

	/*********************** Build Methods  ***********************/
	CFTCMarginReport build();
	
	CFTCMarginReport.CFTCMarginReportBuilder toBuilder();
	
	static CFTCMarginReport.CFTCMarginReportBuilder builder() {
		return new CFTCMarginReport.CFTCMarginReportBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends CFTCMarginReport> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends CFTCMarginReport> getType() {
		return CFTCMarginReport.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processor.processBasic(path.newSubPath("counterparty1"), String.class, getCounterparty1(), this);
		processor.processBasic(path.newSubPath("counterparty2"), String.class, getCounterparty2(), this);
		processor.processBasic(path.newSubPath("counterparty2IdentifierSource"), String.class, getCounterparty2IdentifierSource(), this);
		processor.processBasic(path.newSubPath("submitterIdentifier"), String.class, getSubmitterIdentifier(), this);
		processor.processBasic(path.newSubPath("actionType"), MarginActionEnum.class, getActionType(), this);
		processor.processBasic(path.newSubPath("eventTimestamp"), ZonedDateTime.class, getEventTimestamp(), this);
		processor.processBasic(path.newSubPath("reportingTimestamp"), ZonedDateTime.class, getReportingTimestamp(), this);
		processor.processBasic(path.newSubPath("uniqueSwapIdentifier"), String.class, getUniqueSwapIdentifier(), this);
		processor.processBasic(path.newSubPath("uniqueTransactionIdentifier"), String.class, getUniqueTransactionIdentifier(), this);
		processor.processBasic(path.newSubPath("affiliatedCounterpartyForMarginAndCapitalIndicator"), Boolean.class, getAffiliatedCounterpartyForMarginAndCapitalIndicator(), this);
		processor.processBasic(path.newSubPath("collateralisationCategory"), CollateralisationType3Code__1.class, getCollateralisationCategory(), this);
		processor.processBasic(path.newSubPath("initialMarginCollateralPortfolioCode"), String.class, getInitialMarginCollateralPortfolioCode(), this);
		processor.processBasic(path.newSubPath("portfolioContainingNonReportableComponentIndicator"), Boolean.class, getPortfolioContainingNonReportableComponentIndicator(), this);
		processor.processBasic(path.newSubPath("initialMarginPostedByTheReportingCounterpartyPostHaircut"), BigDecimal.class, getInitialMarginPostedByTheReportingCounterpartyPostHaircut(), this);
		processor.processBasic(path.newSubPath("initialMarginPostedByTheReportingCounterpartyPreHaircut"), BigDecimal.class, getInitialMarginPostedByTheReportingCounterpartyPreHaircut(), this);
		processor.processBasic(path.newSubPath("currencyOfInitialMarginPosted"), ISOCurrencyCodeEnum.class, getCurrencyOfInitialMarginPosted(), this);
		processor.processBasic(path.newSubPath("initialMarginCollectedByTheReportingCounterpartyPostHaircut"), BigDecimal.class, getInitialMarginCollectedByTheReportingCounterpartyPostHaircut(), this);
		processor.processBasic(path.newSubPath("initialMarginCollectedByTheReportingCounterpartyPreHaircut"), BigDecimal.class, getInitialMarginCollectedByTheReportingCounterpartyPreHaircut(), this);
		processor.processBasic(path.newSubPath("currencyOfInitialMarginCollected"), ISOCurrencyCodeEnum.class, getCurrencyOfInitialMarginCollected(), this);
		processor.processBasic(path.newSubPath("variationMarginCollateralPortfolioCode"), String.class, getVariationMarginCollateralPortfolioCode(), this);
		processor.processBasic(path.newSubPath("variationMarginPostedByTheReportingCounterpartyPreHaircut"), BigDecimal.class, getVariationMarginPostedByTheReportingCounterpartyPreHaircut(), this);
		processor.processBasic(path.newSubPath("currencyOfVariationMarginPosted"), ISOCurrencyCodeEnum.class, getCurrencyOfVariationMarginPosted(), this);
		processor.processBasic(path.newSubPath("variationMarginCollectedByTheReportingCounterpartyPreHaircut"), BigDecimal.class, getVariationMarginCollectedByTheReportingCounterpartyPreHaircut(), this);
		processor.processBasic(path.newSubPath("currencyOfVariationMarginCollected"), ISOCurrencyCodeEnum.class, getCurrencyOfVariationMarginCollected(), this);
		processor.processBasic(path.newSubPath("tradeParty1IDType"), PartyIdentifierTypeEnum.class, getTradeParty1IDType(), this);
		processor.processBasic(path.newSubPath("submittingPartyIDType"), PartyIdentifierTypeEnum.class, getSubmittingPartyIDType(), this);
		processor.processBasic(path.newSubPath("usiIDPrefix"), String.class, getUsiIDPrefix(), this);
		processor.processBasic(path.newSubPath("utiIDPrefix"), String.class, getUtiIDPrefix(), this);
		processor.processBasic(path.newSubPath("tradeParty1ReportingDestination"), SupervisoryBodyEnum.class, getTradeParty1ReportingDestination(), this);
		processor.processBasic(path.newSubPath("comment1"), String.class, getComment1(), this);
		processor.processBasic(path.newSubPath("messageID"), String.class, getMessageID(), this);
		processor.processBasic(path.newSubPath("messageType"), String.class, getMessageType(), this);
		processor.processBasic(path.newSubPath("version"), String.class, getVersion(), this);
	}
	

	/*********************** Builder Interface  ***********************/
	interface CFTCMarginReportBuilder extends CFTCMarginReport, RosettaModelObjectBuilder {
		CFTCMarginReport.CFTCMarginReportBuilder setCounterparty1(String counterparty1);
		CFTCMarginReport.CFTCMarginReportBuilder setCounterparty2(String counterparty2);
		CFTCMarginReport.CFTCMarginReportBuilder setCounterparty2IdentifierSource(String counterparty2IdentifierSource);
		CFTCMarginReport.CFTCMarginReportBuilder setSubmitterIdentifier(String submitterIdentifier);
		CFTCMarginReport.CFTCMarginReportBuilder setActionType(MarginActionEnum actionType);
		CFTCMarginReport.CFTCMarginReportBuilder setEventTimestamp(ZonedDateTime eventTimestamp);
		CFTCMarginReport.CFTCMarginReportBuilder setReportingTimestamp(ZonedDateTime reportingTimestamp);
		CFTCMarginReport.CFTCMarginReportBuilder setUniqueSwapIdentifier(String uniqueSwapIdentifier);
		CFTCMarginReport.CFTCMarginReportBuilder setUniqueTransactionIdentifier(String uniqueTransactionIdentifier);
		CFTCMarginReport.CFTCMarginReportBuilder setAffiliatedCounterpartyForMarginAndCapitalIndicator(Boolean affiliatedCounterpartyForMarginAndCapitalIndicator);
		CFTCMarginReport.CFTCMarginReportBuilder setCollateralisationCategory(CollateralisationType3Code__1 collateralisationCategory);
		CFTCMarginReport.CFTCMarginReportBuilder setInitialMarginCollateralPortfolioCode(String initialMarginCollateralPortfolioCode);
		CFTCMarginReport.CFTCMarginReportBuilder setPortfolioContainingNonReportableComponentIndicator(Boolean portfolioContainingNonReportableComponentIndicator);
		CFTCMarginReport.CFTCMarginReportBuilder setInitialMarginPostedByTheReportingCounterpartyPostHaircut(BigDecimal initialMarginPostedByTheReportingCounterpartyPostHaircut);
		CFTCMarginReport.CFTCMarginReportBuilder setInitialMarginPostedByTheReportingCounterpartyPreHaircut(BigDecimal initialMarginPostedByTheReportingCounterpartyPreHaircut);
		CFTCMarginReport.CFTCMarginReportBuilder setCurrencyOfInitialMarginPosted(ISOCurrencyCodeEnum currencyOfInitialMarginPosted);
		CFTCMarginReport.CFTCMarginReportBuilder setInitialMarginCollectedByTheReportingCounterpartyPostHaircut(BigDecimal initialMarginCollectedByTheReportingCounterpartyPostHaircut);
		CFTCMarginReport.CFTCMarginReportBuilder setInitialMarginCollectedByTheReportingCounterpartyPreHaircut(BigDecimal initialMarginCollectedByTheReportingCounterpartyPreHaircut);
		CFTCMarginReport.CFTCMarginReportBuilder setCurrencyOfInitialMarginCollected(ISOCurrencyCodeEnum currencyOfInitialMarginCollected);
		CFTCMarginReport.CFTCMarginReportBuilder setVariationMarginCollateralPortfolioCode(String variationMarginCollateralPortfolioCode);
		CFTCMarginReport.CFTCMarginReportBuilder setVariationMarginPostedByTheReportingCounterpartyPreHaircut(BigDecimal variationMarginPostedByTheReportingCounterpartyPreHaircut);
		CFTCMarginReport.CFTCMarginReportBuilder setCurrencyOfVariationMarginPosted(ISOCurrencyCodeEnum currencyOfVariationMarginPosted);
		CFTCMarginReport.CFTCMarginReportBuilder setVariationMarginCollectedByTheReportingCounterpartyPreHaircut(BigDecimal variationMarginCollectedByTheReportingCounterpartyPreHaircut);
		CFTCMarginReport.CFTCMarginReportBuilder setCurrencyOfVariationMarginCollected(ISOCurrencyCodeEnum currencyOfVariationMarginCollected);
		CFTCMarginReport.CFTCMarginReportBuilder setTradeParty1IDType(PartyIdentifierTypeEnum tradeParty1IDType);
		CFTCMarginReport.CFTCMarginReportBuilder setSubmittingPartyIDType(PartyIdentifierTypeEnum submittingPartyIDType);
		CFTCMarginReport.CFTCMarginReportBuilder setUsiIDPrefix(String usiIDPrefix);
		CFTCMarginReport.CFTCMarginReportBuilder setUtiIDPrefix(String utiIDPrefix);
		CFTCMarginReport.CFTCMarginReportBuilder addTradeParty1ReportingDestination(SupervisoryBodyEnum tradeParty1ReportingDestination);
		CFTCMarginReport.CFTCMarginReportBuilder addTradeParty1ReportingDestination(SupervisoryBodyEnum tradeParty1ReportingDestination, int idx);
		CFTCMarginReport.CFTCMarginReportBuilder addTradeParty1ReportingDestination(List<SupervisoryBodyEnum> tradeParty1ReportingDestination);
		CFTCMarginReport.CFTCMarginReportBuilder setTradeParty1ReportingDestination(List<SupervisoryBodyEnum> tradeParty1ReportingDestination);
		CFTCMarginReport.CFTCMarginReportBuilder setComment1(String comment1);
		CFTCMarginReport.CFTCMarginReportBuilder setMessageID(String messageID);
		CFTCMarginReport.CFTCMarginReportBuilder setMessageType(String messageType);
		CFTCMarginReport.CFTCMarginReportBuilder setVersion(String version);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processor.processBasic(path.newSubPath("counterparty1"), String.class, getCounterparty1(), this);
			processor.processBasic(path.newSubPath("counterparty2"), String.class, getCounterparty2(), this);
			processor.processBasic(path.newSubPath("counterparty2IdentifierSource"), String.class, getCounterparty2IdentifierSource(), this);
			processor.processBasic(path.newSubPath("submitterIdentifier"), String.class, getSubmitterIdentifier(), this);
			processor.processBasic(path.newSubPath("actionType"), MarginActionEnum.class, getActionType(), this);
			processor.processBasic(path.newSubPath("eventTimestamp"), ZonedDateTime.class, getEventTimestamp(), this);
			processor.processBasic(path.newSubPath("reportingTimestamp"), ZonedDateTime.class, getReportingTimestamp(), this);
			processor.processBasic(path.newSubPath("uniqueSwapIdentifier"), String.class, getUniqueSwapIdentifier(), this);
			processor.processBasic(path.newSubPath("uniqueTransactionIdentifier"), String.class, getUniqueTransactionIdentifier(), this);
			processor.processBasic(path.newSubPath("affiliatedCounterpartyForMarginAndCapitalIndicator"), Boolean.class, getAffiliatedCounterpartyForMarginAndCapitalIndicator(), this);
			processor.processBasic(path.newSubPath("collateralisationCategory"), CollateralisationType3Code__1.class, getCollateralisationCategory(), this);
			processor.processBasic(path.newSubPath("initialMarginCollateralPortfolioCode"), String.class, getInitialMarginCollateralPortfolioCode(), this);
			processor.processBasic(path.newSubPath("portfolioContainingNonReportableComponentIndicator"), Boolean.class, getPortfolioContainingNonReportableComponentIndicator(), this);
			processor.processBasic(path.newSubPath("initialMarginPostedByTheReportingCounterpartyPostHaircut"), BigDecimal.class, getInitialMarginPostedByTheReportingCounterpartyPostHaircut(), this);
			processor.processBasic(path.newSubPath("initialMarginPostedByTheReportingCounterpartyPreHaircut"), BigDecimal.class, getInitialMarginPostedByTheReportingCounterpartyPreHaircut(), this);
			processor.processBasic(path.newSubPath("currencyOfInitialMarginPosted"), ISOCurrencyCodeEnum.class, getCurrencyOfInitialMarginPosted(), this);
			processor.processBasic(path.newSubPath("initialMarginCollectedByTheReportingCounterpartyPostHaircut"), BigDecimal.class, getInitialMarginCollectedByTheReportingCounterpartyPostHaircut(), this);
			processor.processBasic(path.newSubPath("initialMarginCollectedByTheReportingCounterpartyPreHaircut"), BigDecimal.class, getInitialMarginCollectedByTheReportingCounterpartyPreHaircut(), this);
			processor.processBasic(path.newSubPath("currencyOfInitialMarginCollected"), ISOCurrencyCodeEnum.class, getCurrencyOfInitialMarginCollected(), this);
			processor.processBasic(path.newSubPath("variationMarginCollateralPortfolioCode"), String.class, getVariationMarginCollateralPortfolioCode(), this);
			processor.processBasic(path.newSubPath("variationMarginPostedByTheReportingCounterpartyPreHaircut"), BigDecimal.class, getVariationMarginPostedByTheReportingCounterpartyPreHaircut(), this);
			processor.processBasic(path.newSubPath("currencyOfVariationMarginPosted"), ISOCurrencyCodeEnum.class, getCurrencyOfVariationMarginPosted(), this);
			processor.processBasic(path.newSubPath("variationMarginCollectedByTheReportingCounterpartyPreHaircut"), BigDecimal.class, getVariationMarginCollectedByTheReportingCounterpartyPreHaircut(), this);
			processor.processBasic(path.newSubPath("currencyOfVariationMarginCollected"), ISOCurrencyCodeEnum.class, getCurrencyOfVariationMarginCollected(), this);
			processor.processBasic(path.newSubPath("tradeParty1IDType"), PartyIdentifierTypeEnum.class, getTradeParty1IDType(), this);
			processor.processBasic(path.newSubPath("submittingPartyIDType"), PartyIdentifierTypeEnum.class, getSubmittingPartyIDType(), this);
			processor.processBasic(path.newSubPath("usiIDPrefix"), String.class, getUsiIDPrefix(), this);
			processor.processBasic(path.newSubPath("utiIDPrefix"), String.class, getUtiIDPrefix(), this);
			processor.processBasic(path.newSubPath("tradeParty1ReportingDestination"), SupervisoryBodyEnum.class, getTradeParty1ReportingDestination(), this);
			processor.processBasic(path.newSubPath("comment1"), String.class, getComment1(), this);
			processor.processBasic(path.newSubPath("messageID"), String.class, getMessageID(), this);
			processor.processBasic(path.newSubPath("messageType"), String.class, getMessageType(), this);
			processor.processBasic(path.newSubPath("version"), String.class, getVersion(), this);
		}
		

		CFTCMarginReport.CFTCMarginReportBuilder prune();
	}

	/*********************** Immutable Implementation of CFTCMarginReport  ***********************/
	class CFTCMarginReportImpl implements CFTCMarginReport {
		private final String counterparty1;
		private final String counterparty2;
		private final String counterparty2IdentifierSource;
		private final String submitterIdentifier;
		private final MarginActionEnum actionType;
		private final ZonedDateTime eventTimestamp;
		private final ZonedDateTime reportingTimestamp;
		private final String uniqueSwapIdentifier;
		private final String uniqueTransactionIdentifier;
		private final Boolean affiliatedCounterpartyForMarginAndCapitalIndicator;
		private final CollateralisationType3Code__1 collateralisationCategory;
		private final String initialMarginCollateralPortfolioCode;
		private final Boolean portfolioContainingNonReportableComponentIndicator;
		private final BigDecimal initialMarginPostedByTheReportingCounterpartyPostHaircut;
		private final BigDecimal initialMarginPostedByTheReportingCounterpartyPreHaircut;
		private final ISOCurrencyCodeEnum currencyOfInitialMarginPosted;
		private final BigDecimal initialMarginCollectedByTheReportingCounterpartyPostHaircut;
		private final BigDecimal initialMarginCollectedByTheReportingCounterpartyPreHaircut;
		private final ISOCurrencyCodeEnum currencyOfInitialMarginCollected;
		private final String variationMarginCollateralPortfolioCode;
		private final BigDecimal variationMarginPostedByTheReportingCounterpartyPreHaircut;
		private final ISOCurrencyCodeEnum currencyOfVariationMarginPosted;
		private final BigDecimal variationMarginCollectedByTheReportingCounterpartyPreHaircut;
		private final ISOCurrencyCodeEnum currencyOfVariationMarginCollected;
		private final PartyIdentifierTypeEnum tradeParty1IDType;
		private final PartyIdentifierTypeEnum submittingPartyIDType;
		private final String usiIDPrefix;
		private final String utiIDPrefix;
		private final List<SupervisoryBodyEnum> tradeParty1ReportingDestination;
		private final String comment1;
		private final String messageID;
		private final String messageType;
		private final String version;
		
		protected CFTCMarginReportImpl(CFTCMarginReport.CFTCMarginReportBuilder builder) {
			this.counterparty1 = builder.getCounterparty1();
			this.counterparty2 = builder.getCounterparty2();
			this.counterparty2IdentifierSource = builder.getCounterparty2IdentifierSource();
			this.submitterIdentifier = builder.getSubmitterIdentifier();
			this.actionType = builder.getActionType();
			this.eventTimestamp = builder.getEventTimestamp();
			this.reportingTimestamp = builder.getReportingTimestamp();
			this.uniqueSwapIdentifier = builder.getUniqueSwapIdentifier();
			this.uniqueTransactionIdentifier = builder.getUniqueTransactionIdentifier();
			this.affiliatedCounterpartyForMarginAndCapitalIndicator = builder.getAffiliatedCounterpartyForMarginAndCapitalIndicator();
			this.collateralisationCategory = builder.getCollateralisationCategory();
			this.initialMarginCollateralPortfolioCode = builder.getInitialMarginCollateralPortfolioCode();
			this.portfolioContainingNonReportableComponentIndicator = builder.getPortfolioContainingNonReportableComponentIndicator();
			this.initialMarginPostedByTheReportingCounterpartyPostHaircut = builder.getInitialMarginPostedByTheReportingCounterpartyPostHaircut();
			this.initialMarginPostedByTheReportingCounterpartyPreHaircut = builder.getInitialMarginPostedByTheReportingCounterpartyPreHaircut();
			this.currencyOfInitialMarginPosted = builder.getCurrencyOfInitialMarginPosted();
			this.initialMarginCollectedByTheReportingCounterpartyPostHaircut = builder.getInitialMarginCollectedByTheReportingCounterpartyPostHaircut();
			this.initialMarginCollectedByTheReportingCounterpartyPreHaircut = builder.getInitialMarginCollectedByTheReportingCounterpartyPreHaircut();
			this.currencyOfInitialMarginCollected = builder.getCurrencyOfInitialMarginCollected();
			this.variationMarginCollateralPortfolioCode = builder.getVariationMarginCollateralPortfolioCode();
			this.variationMarginPostedByTheReportingCounterpartyPreHaircut = builder.getVariationMarginPostedByTheReportingCounterpartyPreHaircut();
			this.currencyOfVariationMarginPosted = builder.getCurrencyOfVariationMarginPosted();
			this.variationMarginCollectedByTheReportingCounterpartyPreHaircut = builder.getVariationMarginCollectedByTheReportingCounterpartyPreHaircut();
			this.currencyOfVariationMarginCollected = builder.getCurrencyOfVariationMarginCollected();
			this.tradeParty1IDType = builder.getTradeParty1IDType();
			this.submittingPartyIDType = builder.getSubmittingPartyIDType();
			this.usiIDPrefix = builder.getUsiIDPrefix();
			this.utiIDPrefix = builder.getUtiIDPrefix();
			this.tradeParty1ReportingDestination = ofNullable(builder.getTradeParty1ReportingDestination()).filter(_l->!_l.isEmpty()).map(ImmutableList::copyOf).orElse(null);
			this.comment1 = builder.getComment1();
			this.messageID = builder.getMessageID();
			this.messageType = builder.getMessageType();
			this.version = builder.getVersion();
		}
		
		@Override
		@RosettaAttribute(value="counterparty1", isRequired=true)
		@RuneAttribute(value="counterparty1", isRequired=true)
		public String getCounterparty1() {
			return counterparty1;
		}
		
		@Override
		@RosettaAttribute(value="counterparty2", isRequired=true)
		@RuneAttribute(value="counterparty2", isRequired=true)
		public String getCounterparty2() {
			return counterparty2;
		}
		
		@Override
		@RosettaAttribute(value="counterparty2IdentifierSource", isRequired=true)
		@RuneAttribute(value="counterparty2IdentifierSource", isRequired=true)
		public String getCounterparty2IdentifierSource() {
			return counterparty2IdentifierSource;
		}
		
		@Override
		@RosettaAttribute(value="submitterIdentifier", isRequired=true)
		@RuneAttribute(value="submitterIdentifier", isRequired=true)
		public String getSubmitterIdentifier() {
			return submitterIdentifier;
		}
		
		@Override
		@RosettaAttribute(value="actionType", isRequired=true)
		@RuneAttribute(value="actionType", isRequired=true)
		public MarginActionEnum getActionType() {
			return actionType;
		}
		
		@Override
		@RosettaAttribute(value="eventTimestamp", isRequired=true)
		@RuneAttribute(value="eventTimestamp", isRequired=true)
		public ZonedDateTime getEventTimestamp() {
			return eventTimestamp;
		}
		
		@Override
		@RosettaAttribute(value="reportingTimestamp", isRequired=true)
		@RuneAttribute(value="reportingTimestamp", isRequired=true)
		public ZonedDateTime getReportingTimestamp() {
			return reportingTimestamp;
		}
		
		@Override
		@RosettaAttribute(value="uniqueSwapIdentifier", isRequired=true)
		@RuneAttribute(value="uniqueSwapIdentifier", isRequired=true)
		public String getUniqueSwapIdentifier() {
			return uniqueSwapIdentifier;
		}
		
		@Override
		@RosettaAttribute("uniqueTransactionIdentifier")
		@RuneAttribute("uniqueTransactionIdentifier")
		public String getUniqueTransactionIdentifier() {
			return uniqueTransactionIdentifier;
		}
		
		@Override
		@RosettaAttribute(value="affiliatedCounterpartyForMarginAndCapitalIndicator", isRequired=true)
		@RuneAttribute(value="affiliatedCounterpartyForMarginAndCapitalIndicator", isRequired=true)
		public Boolean getAffiliatedCounterpartyForMarginAndCapitalIndicator() {
			return affiliatedCounterpartyForMarginAndCapitalIndicator;
		}
		
		@Override
		@RosettaAttribute("collateralisationCategory")
		@RuneAttribute("collateralisationCategory")
		public CollateralisationType3Code__1 getCollateralisationCategory() {
			return collateralisationCategory;
		}
		
		@Override
		@RosettaAttribute("initialMarginCollateralPortfolioCode")
		@RuneAttribute("initialMarginCollateralPortfolioCode")
		public String getInitialMarginCollateralPortfolioCode() {
			return initialMarginCollateralPortfolioCode;
		}
		
		@Override
		@RosettaAttribute(value="portfolioContainingNonReportableComponentIndicator", isRequired=true)
		@RuneAttribute(value="portfolioContainingNonReportableComponentIndicator", isRequired=true)
		public Boolean getPortfolioContainingNonReportableComponentIndicator() {
			return portfolioContainingNonReportableComponentIndicator;
		}
		
		@Override
		@RosettaAttribute("initialMarginPostedByTheReportingCounterpartyPostHaircut")
		@RuneAttribute("initialMarginPostedByTheReportingCounterpartyPostHaircut")
		public BigDecimal getInitialMarginPostedByTheReportingCounterpartyPostHaircut() {
			return initialMarginPostedByTheReportingCounterpartyPostHaircut;
		}
		
		@Override
		@RosettaAttribute("initialMarginPostedByTheReportingCounterpartyPreHaircut")
		@RuneAttribute("initialMarginPostedByTheReportingCounterpartyPreHaircut")
		public BigDecimal getInitialMarginPostedByTheReportingCounterpartyPreHaircut() {
			return initialMarginPostedByTheReportingCounterpartyPreHaircut;
		}
		
		@Override
		@RosettaAttribute("currencyOfInitialMarginPosted")
		@RuneAttribute("currencyOfInitialMarginPosted")
		public ISOCurrencyCodeEnum getCurrencyOfInitialMarginPosted() {
			return currencyOfInitialMarginPosted;
		}
		
		@Override
		@RosettaAttribute("initialMarginCollectedByTheReportingCounterpartyPostHaircut")
		@RuneAttribute("initialMarginCollectedByTheReportingCounterpartyPostHaircut")
		public BigDecimal getInitialMarginCollectedByTheReportingCounterpartyPostHaircut() {
			return initialMarginCollectedByTheReportingCounterpartyPostHaircut;
		}
		
		@Override
		@RosettaAttribute("initialMarginCollectedByTheReportingCounterpartyPreHaircut")
		@RuneAttribute("initialMarginCollectedByTheReportingCounterpartyPreHaircut")
		public BigDecimal getInitialMarginCollectedByTheReportingCounterpartyPreHaircut() {
			return initialMarginCollectedByTheReportingCounterpartyPreHaircut;
		}
		
		@Override
		@RosettaAttribute("currencyOfInitialMarginCollected")
		@RuneAttribute("currencyOfInitialMarginCollected")
		public ISOCurrencyCodeEnum getCurrencyOfInitialMarginCollected() {
			return currencyOfInitialMarginCollected;
		}
		
		@Override
		@RosettaAttribute("variationMarginCollateralPortfolioCode")
		@RuneAttribute("variationMarginCollateralPortfolioCode")
		public String getVariationMarginCollateralPortfolioCode() {
			return variationMarginCollateralPortfolioCode;
		}
		
		@Override
		@RosettaAttribute("variationMarginPostedByTheReportingCounterpartyPreHaircut")
		@RuneAttribute("variationMarginPostedByTheReportingCounterpartyPreHaircut")
		public BigDecimal getVariationMarginPostedByTheReportingCounterpartyPreHaircut() {
			return variationMarginPostedByTheReportingCounterpartyPreHaircut;
		}
		
		@Override
		@RosettaAttribute("currencyOfVariationMarginPosted")
		@RuneAttribute("currencyOfVariationMarginPosted")
		public ISOCurrencyCodeEnum getCurrencyOfVariationMarginPosted() {
			return currencyOfVariationMarginPosted;
		}
		
		@Override
		@RosettaAttribute("variationMarginCollectedByTheReportingCounterpartyPreHaircut")
		@RuneAttribute("variationMarginCollectedByTheReportingCounterpartyPreHaircut")
		public BigDecimal getVariationMarginCollectedByTheReportingCounterpartyPreHaircut() {
			return variationMarginCollectedByTheReportingCounterpartyPreHaircut;
		}
		
		@Override
		@RosettaAttribute("currencyOfVariationMarginCollected")
		@RuneAttribute("currencyOfVariationMarginCollected")
		public ISOCurrencyCodeEnum getCurrencyOfVariationMarginCollected() {
			return currencyOfVariationMarginCollected;
		}
		
		@Override
		@RosettaAttribute(value="tradeParty1IDType", isRequired=true)
		@RuneAttribute(value="tradeParty1IDType", isRequired=true)
		public PartyIdentifierTypeEnum getTradeParty1IDType() {
			return tradeParty1IDType;
		}
		
		@Override
		@RosettaAttribute(value="submittingPartyIDType", isRequired=true)
		@RuneAttribute(value="submittingPartyIDType", isRequired=true)
		public PartyIdentifierTypeEnum getSubmittingPartyIDType() {
			return submittingPartyIDType;
		}
		
		@Override
		@RosettaAttribute("usiIDPrefix")
		@RuneAttribute("usiIDPrefix")
		public String getUsiIDPrefix() {
			return usiIDPrefix;
		}
		
		@Override
		@RosettaAttribute("utiIDPrefix")
		@RuneAttribute("utiIDPrefix")
		public String getUtiIDPrefix() {
			return utiIDPrefix;
		}
		
		@Override
		@RosettaAttribute(value="tradeParty1ReportingDestination", isRequired=true)
		@RuneAttribute(value="tradeParty1ReportingDestination", isRequired=true)
		public List<SupervisoryBodyEnum> getTradeParty1ReportingDestination() {
			return tradeParty1ReportingDestination;
		}
		
		@Override
		@RosettaAttribute(value="comment1", isRequired=true)
		@RuneAttribute(value="comment1", isRequired=true)
		public String getComment1() {
			return comment1;
		}
		
		@Override
		@RosettaAttribute(value="messageID", isRequired=true)
		@RuneAttribute(value="messageID", isRequired=true)
		public String getMessageID() {
			return messageID;
		}
		
		@Override
		@RosettaAttribute(value="messageType", isRequired=true)
		@RuneAttribute(value="messageType", isRequired=true)
		public String getMessageType() {
			return messageType;
		}
		
		@Override
		@RosettaAttribute(value="version", isRequired=true)
		@RuneAttribute(value="version", isRequired=true)
		public String getVersion() {
			return version;
		}
		
		@Override
		public CFTCMarginReport build() {
			return this;
		}
		
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder toBuilder() {
			CFTCMarginReport.CFTCMarginReportBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(CFTCMarginReport.CFTCMarginReportBuilder builder) {
			ofNullable(getCounterparty1()).ifPresent(builder::setCounterparty1);
			ofNullable(getCounterparty2()).ifPresent(builder::setCounterparty2);
			ofNullable(getCounterparty2IdentifierSource()).ifPresent(builder::setCounterparty2IdentifierSource);
			ofNullable(getSubmitterIdentifier()).ifPresent(builder::setSubmitterIdentifier);
			ofNullable(getActionType()).ifPresent(builder::setActionType);
			ofNullable(getEventTimestamp()).ifPresent(builder::setEventTimestamp);
			ofNullable(getReportingTimestamp()).ifPresent(builder::setReportingTimestamp);
			ofNullable(getUniqueSwapIdentifier()).ifPresent(builder::setUniqueSwapIdentifier);
			ofNullable(getUniqueTransactionIdentifier()).ifPresent(builder::setUniqueTransactionIdentifier);
			ofNullable(getAffiliatedCounterpartyForMarginAndCapitalIndicator()).ifPresent(builder::setAffiliatedCounterpartyForMarginAndCapitalIndicator);
			ofNullable(getCollateralisationCategory()).ifPresent(builder::setCollateralisationCategory);
			ofNullable(getInitialMarginCollateralPortfolioCode()).ifPresent(builder::setInitialMarginCollateralPortfolioCode);
			ofNullable(getPortfolioContainingNonReportableComponentIndicator()).ifPresent(builder::setPortfolioContainingNonReportableComponentIndicator);
			ofNullable(getInitialMarginPostedByTheReportingCounterpartyPostHaircut()).ifPresent(builder::setInitialMarginPostedByTheReportingCounterpartyPostHaircut);
			ofNullable(getInitialMarginPostedByTheReportingCounterpartyPreHaircut()).ifPresent(builder::setInitialMarginPostedByTheReportingCounterpartyPreHaircut);
			ofNullable(getCurrencyOfInitialMarginPosted()).ifPresent(builder::setCurrencyOfInitialMarginPosted);
			ofNullable(getInitialMarginCollectedByTheReportingCounterpartyPostHaircut()).ifPresent(builder::setInitialMarginCollectedByTheReportingCounterpartyPostHaircut);
			ofNullable(getInitialMarginCollectedByTheReportingCounterpartyPreHaircut()).ifPresent(builder::setInitialMarginCollectedByTheReportingCounterpartyPreHaircut);
			ofNullable(getCurrencyOfInitialMarginCollected()).ifPresent(builder::setCurrencyOfInitialMarginCollected);
			ofNullable(getVariationMarginCollateralPortfolioCode()).ifPresent(builder::setVariationMarginCollateralPortfolioCode);
			ofNullable(getVariationMarginPostedByTheReportingCounterpartyPreHaircut()).ifPresent(builder::setVariationMarginPostedByTheReportingCounterpartyPreHaircut);
			ofNullable(getCurrencyOfVariationMarginPosted()).ifPresent(builder::setCurrencyOfVariationMarginPosted);
			ofNullable(getVariationMarginCollectedByTheReportingCounterpartyPreHaircut()).ifPresent(builder::setVariationMarginCollectedByTheReportingCounterpartyPreHaircut);
			ofNullable(getCurrencyOfVariationMarginCollected()).ifPresent(builder::setCurrencyOfVariationMarginCollected);
			ofNullable(getTradeParty1IDType()).ifPresent(builder::setTradeParty1IDType);
			ofNullable(getSubmittingPartyIDType()).ifPresent(builder::setSubmittingPartyIDType);
			ofNullable(getUsiIDPrefix()).ifPresent(builder::setUsiIDPrefix);
			ofNullable(getUtiIDPrefix()).ifPresent(builder::setUtiIDPrefix);
			ofNullable(getTradeParty1ReportingDestination()).ifPresent(builder::setTradeParty1ReportingDestination);
			ofNullable(getComment1()).ifPresent(builder::setComment1);
			ofNullable(getMessageID()).ifPresent(builder::setMessageID);
			ofNullable(getMessageType()).ifPresent(builder::setMessageType);
			ofNullable(getVersion()).ifPresent(builder::setVersion);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			CFTCMarginReport _that = getType().cast(o);
		
			if (!Objects.equals(counterparty1, _that.getCounterparty1())) return false;
			if (!Objects.equals(counterparty2, _that.getCounterparty2())) return false;
			if (!Objects.equals(counterparty2IdentifierSource, _that.getCounterparty2IdentifierSource())) return false;
			if (!Objects.equals(submitterIdentifier, _that.getSubmitterIdentifier())) return false;
			if (!Objects.equals(actionType, _that.getActionType())) return false;
			if (!Objects.equals(eventTimestamp, _that.getEventTimestamp())) return false;
			if (!Objects.equals(reportingTimestamp, _that.getReportingTimestamp())) return false;
			if (!Objects.equals(uniqueSwapIdentifier, _that.getUniqueSwapIdentifier())) return false;
			if (!Objects.equals(uniqueTransactionIdentifier, _that.getUniqueTransactionIdentifier())) return false;
			if (!Objects.equals(affiliatedCounterpartyForMarginAndCapitalIndicator, _that.getAffiliatedCounterpartyForMarginAndCapitalIndicator())) return false;
			if (!Objects.equals(collateralisationCategory, _that.getCollateralisationCategory())) return false;
			if (!Objects.equals(initialMarginCollateralPortfolioCode, _that.getInitialMarginCollateralPortfolioCode())) return false;
			if (!Objects.equals(portfolioContainingNonReportableComponentIndicator, _that.getPortfolioContainingNonReportableComponentIndicator())) return false;
			if (!Objects.equals(initialMarginPostedByTheReportingCounterpartyPostHaircut, _that.getInitialMarginPostedByTheReportingCounterpartyPostHaircut())) return false;
			if (!Objects.equals(initialMarginPostedByTheReportingCounterpartyPreHaircut, _that.getInitialMarginPostedByTheReportingCounterpartyPreHaircut())) return false;
			if (!Objects.equals(currencyOfInitialMarginPosted, _that.getCurrencyOfInitialMarginPosted())) return false;
			if (!Objects.equals(initialMarginCollectedByTheReportingCounterpartyPostHaircut, _that.getInitialMarginCollectedByTheReportingCounterpartyPostHaircut())) return false;
			if (!Objects.equals(initialMarginCollectedByTheReportingCounterpartyPreHaircut, _that.getInitialMarginCollectedByTheReportingCounterpartyPreHaircut())) return false;
			if (!Objects.equals(currencyOfInitialMarginCollected, _that.getCurrencyOfInitialMarginCollected())) return false;
			if (!Objects.equals(variationMarginCollateralPortfolioCode, _that.getVariationMarginCollateralPortfolioCode())) return false;
			if (!Objects.equals(variationMarginPostedByTheReportingCounterpartyPreHaircut, _that.getVariationMarginPostedByTheReportingCounterpartyPreHaircut())) return false;
			if (!Objects.equals(currencyOfVariationMarginPosted, _that.getCurrencyOfVariationMarginPosted())) return false;
			if (!Objects.equals(variationMarginCollectedByTheReportingCounterpartyPreHaircut, _that.getVariationMarginCollectedByTheReportingCounterpartyPreHaircut())) return false;
			if (!Objects.equals(currencyOfVariationMarginCollected, _that.getCurrencyOfVariationMarginCollected())) return false;
			if (!Objects.equals(tradeParty1IDType, _that.getTradeParty1IDType())) return false;
			if (!Objects.equals(submittingPartyIDType, _that.getSubmittingPartyIDType())) return false;
			if (!Objects.equals(usiIDPrefix, _that.getUsiIDPrefix())) return false;
			if (!Objects.equals(utiIDPrefix, _that.getUtiIDPrefix())) return false;
			if (!ListEquals.listEquals(tradeParty1ReportingDestination, _that.getTradeParty1ReportingDestination())) return false;
			if (!Objects.equals(comment1, _that.getComment1())) return false;
			if (!Objects.equals(messageID, _that.getMessageID())) return false;
			if (!Objects.equals(messageType, _that.getMessageType())) return false;
			if (!Objects.equals(version, _that.getVersion())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (counterparty1 != null ? counterparty1.hashCode() : 0);
			_result = 31 * _result + (counterparty2 != null ? counterparty2.hashCode() : 0);
			_result = 31 * _result + (counterparty2IdentifierSource != null ? counterparty2IdentifierSource.hashCode() : 0);
			_result = 31 * _result + (submitterIdentifier != null ? submitterIdentifier.hashCode() : 0);
			_result = 31 * _result + (actionType != null ? actionType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (eventTimestamp != null ? eventTimestamp.hashCode() : 0);
			_result = 31 * _result + (reportingTimestamp != null ? reportingTimestamp.hashCode() : 0);
			_result = 31 * _result + (uniqueSwapIdentifier != null ? uniqueSwapIdentifier.hashCode() : 0);
			_result = 31 * _result + (uniqueTransactionIdentifier != null ? uniqueTransactionIdentifier.hashCode() : 0);
			_result = 31 * _result + (affiliatedCounterpartyForMarginAndCapitalIndicator != null ? affiliatedCounterpartyForMarginAndCapitalIndicator.hashCode() : 0);
			_result = 31 * _result + (collateralisationCategory != null ? collateralisationCategory.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (initialMarginCollateralPortfolioCode != null ? initialMarginCollateralPortfolioCode.hashCode() : 0);
			_result = 31 * _result + (portfolioContainingNonReportableComponentIndicator != null ? portfolioContainingNonReportableComponentIndicator.hashCode() : 0);
			_result = 31 * _result + (initialMarginPostedByTheReportingCounterpartyPostHaircut != null ? initialMarginPostedByTheReportingCounterpartyPostHaircut.hashCode() : 0);
			_result = 31 * _result + (initialMarginPostedByTheReportingCounterpartyPreHaircut != null ? initialMarginPostedByTheReportingCounterpartyPreHaircut.hashCode() : 0);
			_result = 31 * _result + (currencyOfInitialMarginPosted != null ? currencyOfInitialMarginPosted.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (initialMarginCollectedByTheReportingCounterpartyPostHaircut != null ? initialMarginCollectedByTheReportingCounterpartyPostHaircut.hashCode() : 0);
			_result = 31 * _result + (initialMarginCollectedByTheReportingCounterpartyPreHaircut != null ? initialMarginCollectedByTheReportingCounterpartyPreHaircut.hashCode() : 0);
			_result = 31 * _result + (currencyOfInitialMarginCollected != null ? currencyOfInitialMarginCollected.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (variationMarginCollateralPortfolioCode != null ? variationMarginCollateralPortfolioCode.hashCode() : 0);
			_result = 31 * _result + (variationMarginPostedByTheReportingCounterpartyPreHaircut != null ? variationMarginPostedByTheReportingCounterpartyPreHaircut.hashCode() : 0);
			_result = 31 * _result + (currencyOfVariationMarginPosted != null ? currencyOfVariationMarginPosted.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (variationMarginCollectedByTheReportingCounterpartyPreHaircut != null ? variationMarginCollectedByTheReportingCounterpartyPreHaircut.hashCode() : 0);
			_result = 31 * _result + (currencyOfVariationMarginCollected != null ? currencyOfVariationMarginCollected.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (tradeParty1IDType != null ? tradeParty1IDType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (submittingPartyIDType != null ? submittingPartyIDType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (usiIDPrefix != null ? usiIDPrefix.hashCode() : 0);
			_result = 31 * _result + (utiIDPrefix != null ? utiIDPrefix.hashCode() : 0);
			_result = 31 * _result + (tradeParty1ReportingDestination != null ? tradeParty1ReportingDestination.stream().map(Object::getClass).map(Class::getName).mapToInt(String::hashCode).sum() : 0);
			_result = 31 * _result + (comment1 != null ? comment1.hashCode() : 0);
			_result = 31 * _result + (messageID != null ? messageID.hashCode() : 0);
			_result = 31 * _result + (messageType != null ? messageType.hashCode() : 0);
			_result = 31 * _result + (version != null ? version.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "CFTCMarginReport {" +
				"counterparty1=" + this.counterparty1 + ", " +
				"counterparty2=" + this.counterparty2 + ", " +
				"counterparty2IdentifierSource=" + this.counterparty2IdentifierSource + ", " +
				"submitterIdentifier=" + this.submitterIdentifier + ", " +
				"actionType=" + this.actionType + ", " +
				"eventTimestamp=" + this.eventTimestamp + ", " +
				"reportingTimestamp=" + this.reportingTimestamp + ", " +
				"uniqueSwapIdentifier=" + this.uniqueSwapIdentifier + ", " +
				"uniqueTransactionIdentifier=" + this.uniqueTransactionIdentifier + ", " +
				"affiliatedCounterpartyForMarginAndCapitalIndicator=" + this.affiliatedCounterpartyForMarginAndCapitalIndicator + ", " +
				"collateralisationCategory=" + this.collateralisationCategory + ", " +
				"initialMarginCollateralPortfolioCode=" + this.initialMarginCollateralPortfolioCode + ", " +
				"portfolioContainingNonReportableComponentIndicator=" + this.portfolioContainingNonReportableComponentIndicator + ", " +
				"initialMarginPostedByTheReportingCounterpartyPostHaircut=" + this.initialMarginPostedByTheReportingCounterpartyPostHaircut + ", " +
				"initialMarginPostedByTheReportingCounterpartyPreHaircut=" + this.initialMarginPostedByTheReportingCounterpartyPreHaircut + ", " +
				"currencyOfInitialMarginPosted=" + this.currencyOfInitialMarginPosted + ", " +
				"initialMarginCollectedByTheReportingCounterpartyPostHaircut=" + this.initialMarginCollectedByTheReportingCounterpartyPostHaircut + ", " +
				"initialMarginCollectedByTheReportingCounterpartyPreHaircut=" + this.initialMarginCollectedByTheReportingCounterpartyPreHaircut + ", " +
				"currencyOfInitialMarginCollected=" + this.currencyOfInitialMarginCollected + ", " +
				"variationMarginCollateralPortfolioCode=" + this.variationMarginCollateralPortfolioCode + ", " +
				"variationMarginPostedByTheReportingCounterpartyPreHaircut=" + this.variationMarginPostedByTheReportingCounterpartyPreHaircut + ", " +
				"currencyOfVariationMarginPosted=" + this.currencyOfVariationMarginPosted + ", " +
				"variationMarginCollectedByTheReportingCounterpartyPreHaircut=" + this.variationMarginCollectedByTheReportingCounterpartyPreHaircut + ", " +
				"currencyOfVariationMarginCollected=" + this.currencyOfVariationMarginCollected + ", " +
				"tradeParty1IDType=" + this.tradeParty1IDType + ", " +
				"submittingPartyIDType=" + this.submittingPartyIDType + ", " +
				"usiIDPrefix=" + this.usiIDPrefix + ", " +
				"utiIDPrefix=" + this.utiIDPrefix + ", " +
				"tradeParty1ReportingDestination=" + this.tradeParty1ReportingDestination + ", " +
				"comment1=" + this.comment1 + ", " +
				"messageID=" + this.messageID + ", " +
				"messageType=" + this.messageType + ", " +
				"version=" + this.version +
			'}';
		}
	}

	/*********************** Builder Implementation of CFTCMarginReport  ***********************/
	class CFTCMarginReportBuilderImpl implements CFTCMarginReport.CFTCMarginReportBuilder {
	
		protected String counterparty1;
		protected String counterparty2;
		protected String counterparty2IdentifierSource;
		protected String submitterIdentifier;
		protected MarginActionEnum actionType;
		protected ZonedDateTime eventTimestamp;
		protected ZonedDateTime reportingTimestamp;
		protected String uniqueSwapIdentifier;
		protected String uniqueTransactionIdentifier;
		protected Boolean affiliatedCounterpartyForMarginAndCapitalIndicator;
		protected CollateralisationType3Code__1 collateralisationCategory;
		protected String initialMarginCollateralPortfolioCode;
		protected Boolean portfolioContainingNonReportableComponentIndicator;
		protected BigDecimal initialMarginPostedByTheReportingCounterpartyPostHaircut;
		protected BigDecimal initialMarginPostedByTheReportingCounterpartyPreHaircut;
		protected ISOCurrencyCodeEnum currencyOfInitialMarginPosted;
		protected BigDecimal initialMarginCollectedByTheReportingCounterpartyPostHaircut;
		protected BigDecimal initialMarginCollectedByTheReportingCounterpartyPreHaircut;
		protected ISOCurrencyCodeEnum currencyOfInitialMarginCollected;
		protected String variationMarginCollateralPortfolioCode;
		protected BigDecimal variationMarginPostedByTheReportingCounterpartyPreHaircut;
		protected ISOCurrencyCodeEnum currencyOfVariationMarginPosted;
		protected BigDecimal variationMarginCollectedByTheReportingCounterpartyPreHaircut;
		protected ISOCurrencyCodeEnum currencyOfVariationMarginCollected;
		protected PartyIdentifierTypeEnum tradeParty1IDType;
		protected PartyIdentifierTypeEnum submittingPartyIDType;
		protected String usiIDPrefix;
		protected String utiIDPrefix;
		protected List<SupervisoryBodyEnum> tradeParty1ReportingDestination = new ArrayList<>();
		protected String comment1;
		protected String messageID;
		protected String messageType;
		protected String version;
		
		@Override
		@RosettaAttribute(value="counterparty1", isRequired=true)
		@RuneAttribute(value="counterparty1", isRequired=true)
		public String getCounterparty1() {
			return counterparty1;
		}
		
		@Override
		@RosettaAttribute(value="counterparty2", isRequired=true)
		@RuneAttribute(value="counterparty2", isRequired=true)
		public String getCounterparty2() {
			return counterparty2;
		}
		
		@Override
		@RosettaAttribute(value="counterparty2IdentifierSource", isRequired=true)
		@RuneAttribute(value="counterparty2IdentifierSource", isRequired=true)
		public String getCounterparty2IdentifierSource() {
			return counterparty2IdentifierSource;
		}
		
		@Override
		@RosettaAttribute(value="submitterIdentifier", isRequired=true)
		@RuneAttribute(value="submitterIdentifier", isRequired=true)
		public String getSubmitterIdentifier() {
			return submitterIdentifier;
		}
		
		@Override
		@RosettaAttribute(value="actionType", isRequired=true)
		@RuneAttribute(value="actionType", isRequired=true)
		public MarginActionEnum getActionType() {
			return actionType;
		}
		
		@Override
		@RosettaAttribute(value="eventTimestamp", isRequired=true)
		@RuneAttribute(value="eventTimestamp", isRequired=true)
		public ZonedDateTime getEventTimestamp() {
			return eventTimestamp;
		}
		
		@Override
		@RosettaAttribute(value="reportingTimestamp", isRequired=true)
		@RuneAttribute(value="reportingTimestamp", isRequired=true)
		public ZonedDateTime getReportingTimestamp() {
			return reportingTimestamp;
		}
		
		@Override
		@RosettaAttribute(value="uniqueSwapIdentifier", isRequired=true)
		@RuneAttribute(value="uniqueSwapIdentifier", isRequired=true)
		public String getUniqueSwapIdentifier() {
			return uniqueSwapIdentifier;
		}
		
		@Override
		@RosettaAttribute("uniqueTransactionIdentifier")
		@RuneAttribute("uniqueTransactionIdentifier")
		public String getUniqueTransactionIdentifier() {
			return uniqueTransactionIdentifier;
		}
		
		@Override
		@RosettaAttribute(value="affiliatedCounterpartyForMarginAndCapitalIndicator", isRequired=true)
		@RuneAttribute(value="affiliatedCounterpartyForMarginAndCapitalIndicator", isRequired=true)
		public Boolean getAffiliatedCounterpartyForMarginAndCapitalIndicator() {
			return affiliatedCounterpartyForMarginAndCapitalIndicator;
		}
		
		@Override
		@RosettaAttribute("collateralisationCategory")
		@RuneAttribute("collateralisationCategory")
		public CollateralisationType3Code__1 getCollateralisationCategory() {
			return collateralisationCategory;
		}
		
		@Override
		@RosettaAttribute("initialMarginCollateralPortfolioCode")
		@RuneAttribute("initialMarginCollateralPortfolioCode")
		public String getInitialMarginCollateralPortfolioCode() {
			return initialMarginCollateralPortfolioCode;
		}
		
		@Override
		@RosettaAttribute(value="portfolioContainingNonReportableComponentIndicator", isRequired=true)
		@RuneAttribute(value="portfolioContainingNonReportableComponentIndicator", isRequired=true)
		public Boolean getPortfolioContainingNonReportableComponentIndicator() {
			return portfolioContainingNonReportableComponentIndicator;
		}
		
		@Override
		@RosettaAttribute("initialMarginPostedByTheReportingCounterpartyPostHaircut")
		@RuneAttribute("initialMarginPostedByTheReportingCounterpartyPostHaircut")
		public BigDecimal getInitialMarginPostedByTheReportingCounterpartyPostHaircut() {
			return initialMarginPostedByTheReportingCounterpartyPostHaircut;
		}
		
		@Override
		@RosettaAttribute("initialMarginPostedByTheReportingCounterpartyPreHaircut")
		@RuneAttribute("initialMarginPostedByTheReportingCounterpartyPreHaircut")
		public BigDecimal getInitialMarginPostedByTheReportingCounterpartyPreHaircut() {
			return initialMarginPostedByTheReportingCounterpartyPreHaircut;
		}
		
		@Override
		@RosettaAttribute("currencyOfInitialMarginPosted")
		@RuneAttribute("currencyOfInitialMarginPosted")
		public ISOCurrencyCodeEnum getCurrencyOfInitialMarginPosted() {
			return currencyOfInitialMarginPosted;
		}
		
		@Override
		@RosettaAttribute("initialMarginCollectedByTheReportingCounterpartyPostHaircut")
		@RuneAttribute("initialMarginCollectedByTheReportingCounterpartyPostHaircut")
		public BigDecimal getInitialMarginCollectedByTheReportingCounterpartyPostHaircut() {
			return initialMarginCollectedByTheReportingCounterpartyPostHaircut;
		}
		
		@Override
		@RosettaAttribute("initialMarginCollectedByTheReportingCounterpartyPreHaircut")
		@RuneAttribute("initialMarginCollectedByTheReportingCounterpartyPreHaircut")
		public BigDecimal getInitialMarginCollectedByTheReportingCounterpartyPreHaircut() {
			return initialMarginCollectedByTheReportingCounterpartyPreHaircut;
		}
		
		@Override
		@RosettaAttribute("currencyOfInitialMarginCollected")
		@RuneAttribute("currencyOfInitialMarginCollected")
		public ISOCurrencyCodeEnum getCurrencyOfInitialMarginCollected() {
			return currencyOfInitialMarginCollected;
		}
		
		@Override
		@RosettaAttribute("variationMarginCollateralPortfolioCode")
		@RuneAttribute("variationMarginCollateralPortfolioCode")
		public String getVariationMarginCollateralPortfolioCode() {
			return variationMarginCollateralPortfolioCode;
		}
		
		@Override
		@RosettaAttribute("variationMarginPostedByTheReportingCounterpartyPreHaircut")
		@RuneAttribute("variationMarginPostedByTheReportingCounterpartyPreHaircut")
		public BigDecimal getVariationMarginPostedByTheReportingCounterpartyPreHaircut() {
			return variationMarginPostedByTheReportingCounterpartyPreHaircut;
		}
		
		@Override
		@RosettaAttribute("currencyOfVariationMarginPosted")
		@RuneAttribute("currencyOfVariationMarginPosted")
		public ISOCurrencyCodeEnum getCurrencyOfVariationMarginPosted() {
			return currencyOfVariationMarginPosted;
		}
		
		@Override
		@RosettaAttribute("variationMarginCollectedByTheReportingCounterpartyPreHaircut")
		@RuneAttribute("variationMarginCollectedByTheReportingCounterpartyPreHaircut")
		public BigDecimal getVariationMarginCollectedByTheReportingCounterpartyPreHaircut() {
			return variationMarginCollectedByTheReportingCounterpartyPreHaircut;
		}
		
		@Override
		@RosettaAttribute("currencyOfVariationMarginCollected")
		@RuneAttribute("currencyOfVariationMarginCollected")
		public ISOCurrencyCodeEnum getCurrencyOfVariationMarginCollected() {
			return currencyOfVariationMarginCollected;
		}
		
		@Override
		@RosettaAttribute(value="tradeParty1IDType", isRequired=true)
		@RuneAttribute(value="tradeParty1IDType", isRequired=true)
		public PartyIdentifierTypeEnum getTradeParty1IDType() {
			return tradeParty1IDType;
		}
		
		@Override
		@RosettaAttribute(value="submittingPartyIDType", isRequired=true)
		@RuneAttribute(value="submittingPartyIDType", isRequired=true)
		public PartyIdentifierTypeEnum getSubmittingPartyIDType() {
			return submittingPartyIDType;
		}
		
		@Override
		@RosettaAttribute("usiIDPrefix")
		@RuneAttribute("usiIDPrefix")
		public String getUsiIDPrefix() {
			return usiIDPrefix;
		}
		
		@Override
		@RosettaAttribute("utiIDPrefix")
		@RuneAttribute("utiIDPrefix")
		public String getUtiIDPrefix() {
			return utiIDPrefix;
		}
		
		@Override
		@RosettaAttribute(value="tradeParty1ReportingDestination", isRequired=true)
		@RuneAttribute(value="tradeParty1ReportingDestination", isRequired=true)
		public List<SupervisoryBodyEnum> getTradeParty1ReportingDestination() {
			return tradeParty1ReportingDestination;
		}
		
		@Override
		@RosettaAttribute(value="comment1", isRequired=true)
		@RuneAttribute(value="comment1", isRequired=true)
		public String getComment1() {
			return comment1;
		}
		
		@Override
		@RosettaAttribute(value="messageID", isRequired=true)
		@RuneAttribute(value="messageID", isRequired=true)
		public String getMessageID() {
			return messageID;
		}
		
		@Override
		@RosettaAttribute(value="messageType", isRequired=true)
		@RuneAttribute(value="messageType", isRequired=true)
		public String getMessageType() {
			return messageType;
		}
		
		@Override
		@RosettaAttribute(value="version", isRequired=true)
		@RuneAttribute(value="version", isRequired=true)
		public String getVersion() {
			return version;
		}
		
		@RosettaAttribute(value="counterparty1", isRequired=true)
		@RuneAttribute(value="counterparty1", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setCounterparty1(String _counterparty1) {
			this.counterparty1 = _counterparty1 == null ? null : _counterparty1;
			return this;
		}
		
		@RosettaAttribute(value="counterparty2", isRequired=true)
		@RuneAttribute(value="counterparty2", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setCounterparty2(String _counterparty2) {
			this.counterparty2 = _counterparty2 == null ? null : _counterparty2;
			return this;
		}
		
		@RosettaAttribute(value="counterparty2IdentifierSource", isRequired=true)
		@RuneAttribute(value="counterparty2IdentifierSource", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setCounterparty2IdentifierSource(String _counterparty2IdentifierSource) {
			this.counterparty2IdentifierSource = _counterparty2IdentifierSource == null ? null : _counterparty2IdentifierSource;
			return this;
		}
		
		@RosettaAttribute(value="submitterIdentifier", isRequired=true)
		@RuneAttribute(value="submitterIdentifier", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setSubmitterIdentifier(String _submitterIdentifier) {
			this.submitterIdentifier = _submitterIdentifier == null ? null : _submitterIdentifier;
			return this;
		}
		
		@RosettaAttribute(value="actionType", isRequired=true)
		@RuneAttribute(value="actionType", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setActionType(MarginActionEnum _actionType) {
			this.actionType = _actionType == null ? null : _actionType;
			return this;
		}
		
		@RosettaAttribute(value="eventTimestamp", isRequired=true)
		@RuneAttribute(value="eventTimestamp", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setEventTimestamp(ZonedDateTime _eventTimestamp) {
			this.eventTimestamp = _eventTimestamp == null ? null : _eventTimestamp;
			return this;
		}
		
		@RosettaAttribute(value="reportingTimestamp", isRequired=true)
		@RuneAttribute(value="reportingTimestamp", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setReportingTimestamp(ZonedDateTime _reportingTimestamp) {
			this.reportingTimestamp = _reportingTimestamp == null ? null : _reportingTimestamp;
			return this;
		}
		
		@RosettaAttribute(value="uniqueSwapIdentifier", isRequired=true)
		@RuneAttribute(value="uniqueSwapIdentifier", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setUniqueSwapIdentifier(String _uniqueSwapIdentifier) {
			this.uniqueSwapIdentifier = _uniqueSwapIdentifier == null ? null : _uniqueSwapIdentifier;
			return this;
		}
		
		@RosettaAttribute("uniqueTransactionIdentifier")
		@RuneAttribute("uniqueTransactionIdentifier")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setUniqueTransactionIdentifier(String _uniqueTransactionIdentifier) {
			this.uniqueTransactionIdentifier = _uniqueTransactionIdentifier == null ? null : _uniqueTransactionIdentifier;
			return this;
		}
		
		@RosettaAttribute(value="affiliatedCounterpartyForMarginAndCapitalIndicator", isRequired=true)
		@RuneAttribute(value="affiliatedCounterpartyForMarginAndCapitalIndicator", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setAffiliatedCounterpartyForMarginAndCapitalIndicator(Boolean _affiliatedCounterpartyForMarginAndCapitalIndicator) {
			this.affiliatedCounterpartyForMarginAndCapitalIndicator = _affiliatedCounterpartyForMarginAndCapitalIndicator == null ? null : _affiliatedCounterpartyForMarginAndCapitalIndicator;
			return this;
		}
		
		@RosettaAttribute("collateralisationCategory")
		@RuneAttribute("collateralisationCategory")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setCollateralisationCategory(CollateralisationType3Code__1 _collateralisationCategory) {
			this.collateralisationCategory = _collateralisationCategory == null ? null : _collateralisationCategory;
			return this;
		}
		
		@RosettaAttribute("initialMarginCollateralPortfolioCode")
		@RuneAttribute("initialMarginCollateralPortfolioCode")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setInitialMarginCollateralPortfolioCode(String _initialMarginCollateralPortfolioCode) {
			this.initialMarginCollateralPortfolioCode = _initialMarginCollateralPortfolioCode == null ? null : _initialMarginCollateralPortfolioCode;
			return this;
		}
		
		@RosettaAttribute(value="portfolioContainingNonReportableComponentIndicator", isRequired=true)
		@RuneAttribute(value="portfolioContainingNonReportableComponentIndicator", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setPortfolioContainingNonReportableComponentIndicator(Boolean _portfolioContainingNonReportableComponentIndicator) {
			this.portfolioContainingNonReportableComponentIndicator = _portfolioContainingNonReportableComponentIndicator == null ? null : _portfolioContainingNonReportableComponentIndicator;
			return this;
		}
		
		@RosettaAttribute("initialMarginPostedByTheReportingCounterpartyPostHaircut")
		@RuneAttribute("initialMarginPostedByTheReportingCounterpartyPostHaircut")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setInitialMarginPostedByTheReportingCounterpartyPostHaircut(BigDecimal _initialMarginPostedByTheReportingCounterpartyPostHaircut) {
			this.initialMarginPostedByTheReportingCounterpartyPostHaircut = _initialMarginPostedByTheReportingCounterpartyPostHaircut == null ? null : _initialMarginPostedByTheReportingCounterpartyPostHaircut;
			return this;
		}
		
		@RosettaAttribute("initialMarginPostedByTheReportingCounterpartyPreHaircut")
		@RuneAttribute("initialMarginPostedByTheReportingCounterpartyPreHaircut")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setInitialMarginPostedByTheReportingCounterpartyPreHaircut(BigDecimal _initialMarginPostedByTheReportingCounterpartyPreHaircut) {
			this.initialMarginPostedByTheReportingCounterpartyPreHaircut = _initialMarginPostedByTheReportingCounterpartyPreHaircut == null ? null : _initialMarginPostedByTheReportingCounterpartyPreHaircut;
			return this;
		}
		
		@RosettaAttribute("currencyOfInitialMarginPosted")
		@RuneAttribute("currencyOfInitialMarginPosted")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setCurrencyOfInitialMarginPosted(ISOCurrencyCodeEnum _currencyOfInitialMarginPosted) {
			this.currencyOfInitialMarginPosted = _currencyOfInitialMarginPosted == null ? null : _currencyOfInitialMarginPosted;
			return this;
		}
		
		@RosettaAttribute("initialMarginCollectedByTheReportingCounterpartyPostHaircut")
		@RuneAttribute("initialMarginCollectedByTheReportingCounterpartyPostHaircut")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setInitialMarginCollectedByTheReportingCounterpartyPostHaircut(BigDecimal _initialMarginCollectedByTheReportingCounterpartyPostHaircut) {
			this.initialMarginCollectedByTheReportingCounterpartyPostHaircut = _initialMarginCollectedByTheReportingCounterpartyPostHaircut == null ? null : _initialMarginCollectedByTheReportingCounterpartyPostHaircut;
			return this;
		}
		
		@RosettaAttribute("initialMarginCollectedByTheReportingCounterpartyPreHaircut")
		@RuneAttribute("initialMarginCollectedByTheReportingCounterpartyPreHaircut")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setInitialMarginCollectedByTheReportingCounterpartyPreHaircut(BigDecimal _initialMarginCollectedByTheReportingCounterpartyPreHaircut) {
			this.initialMarginCollectedByTheReportingCounterpartyPreHaircut = _initialMarginCollectedByTheReportingCounterpartyPreHaircut == null ? null : _initialMarginCollectedByTheReportingCounterpartyPreHaircut;
			return this;
		}
		
		@RosettaAttribute("currencyOfInitialMarginCollected")
		@RuneAttribute("currencyOfInitialMarginCollected")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setCurrencyOfInitialMarginCollected(ISOCurrencyCodeEnum _currencyOfInitialMarginCollected) {
			this.currencyOfInitialMarginCollected = _currencyOfInitialMarginCollected == null ? null : _currencyOfInitialMarginCollected;
			return this;
		}
		
		@RosettaAttribute("variationMarginCollateralPortfolioCode")
		@RuneAttribute("variationMarginCollateralPortfolioCode")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setVariationMarginCollateralPortfolioCode(String _variationMarginCollateralPortfolioCode) {
			this.variationMarginCollateralPortfolioCode = _variationMarginCollateralPortfolioCode == null ? null : _variationMarginCollateralPortfolioCode;
			return this;
		}
		
		@RosettaAttribute("variationMarginPostedByTheReportingCounterpartyPreHaircut")
		@RuneAttribute("variationMarginPostedByTheReportingCounterpartyPreHaircut")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setVariationMarginPostedByTheReportingCounterpartyPreHaircut(BigDecimal _variationMarginPostedByTheReportingCounterpartyPreHaircut) {
			this.variationMarginPostedByTheReportingCounterpartyPreHaircut = _variationMarginPostedByTheReportingCounterpartyPreHaircut == null ? null : _variationMarginPostedByTheReportingCounterpartyPreHaircut;
			return this;
		}
		
		@RosettaAttribute("currencyOfVariationMarginPosted")
		@RuneAttribute("currencyOfVariationMarginPosted")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setCurrencyOfVariationMarginPosted(ISOCurrencyCodeEnum _currencyOfVariationMarginPosted) {
			this.currencyOfVariationMarginPosted = _currencyOfVariationMarginPosted == null ? null : _currencyOfVariationMarginPosted;
			return this;
		}
		
		@RosettaAttribute("variationMarginCollectedByTheReportingCounterpartyPreHaircut")
		@RuneAttribute("variationMarginCollectedByTheReportingCounterpartyPreHaircut")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setVariationMarginCollectedByTheReportingCounterpartyPreHaircut(BigDecimal _variationMarginCollectedByTheReportingCounterpartyPreHaircut) {
			this.variationMarginCollectedByTheReportingCounterpartyPreHaircut = _variationMarginCollectedByTheReportingCounterpartyPreHaircut == null ? null : _variationMarginCollectedByTheReportingCounterpartyPreHaircut;
			return this;
		}
		
		@RosettaAttribute("currencyOfVariationMarginCollected")
		@RuneAttribute("currencyOfVariationMarginCollected")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setCurrencyOfVariationMarginCollected(ISOCurrencyCodeEnum _currencyOfVariationMarginCollected) {
			this.currencyOfVariationMarginCollected = _currencyOfVariationMarginCollected == null ? null : _currencyOfVariationMarginCollected;
			return this;
		}
		
		@RosettaAttribute(value="tradeParty1IDType", isRequired=true)
		@RuneAttribute(value="tradeParty1IDType", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setTradeParty1IDType(PartyIdentifierTypeEnum _tradeParty1IDType) {
			this.tradeParty1IDType = _tradeParty1IDType == null ? null : _tradeParty1IDType;
			return this;
		}
		
		@RosettaAttribute(value="submittingPartyIDType", isRequired=true)
		@RuneAttribute(value="submittingPartyIDType", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setSubmittingPartyIDType(PartyIdentifierTypeEnum _submittingPartyIDType) {
			this.submittingPartyIDType = _submittingPartyIDType == null ? null : _submittingPartyIDType;
			return this;
		}
		
		@RosettaAttribute("usiIDPrefix")
		@RuneAttribute("usiIDPrefix")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setUsiIDPrefix(String _usiIDPrefix) {
			this.usiIDPrefix = _usiIDPrefix == null ? null : _usiIDPrefix;
			return this;
		}
		
		@RosettaAttribute("utiIDPrefix")
		@RuneAttribute("utiIDPrefix")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setUtiIDPrefix(String _utiIDPrefix) {
			this.utiIDPrefix = _utiIDPrefix == null ? null : _utiIDPrefix;
			return this;
		}
		
		@RosettaAttribute(value="tradeParty1ReportingDestination", isRequired=true)
		@RuneAttribute(value="tradeParty1ReportingDestination", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder addTradeParty1ReportingDestination(SupervisoryBodyEnum _tradeParty1ReportingDestination) {
			if (_tradeParty1ReportingDestination != null) {
				this.tradeParty1ReportingDestination.add(_tradeParty1ReportingDestination);
			}
			return this;
		}
		
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder addTradeParty1ReportingDestination(SupervisoryBodyEnum _tradeParty1ReportingDestination, int idx) {
			getIndex(this.tradeParty1ReportingDestination, idx, () -> _tradeParty1ReportingDestination);
			return this;
		}
		
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder addTradeParty1ReportingDestination(List<SupervisoryBodyEnum> tradeParty1ReportingDestinations) {
			if (tradeParty1ReportingDestinations != null) {
				for (final SupervisoryBodyEnum toAdd : tradeParty1ReportingDestinations) {
					this.tradeParty1ReportingDestination.add(toAdd);
				}
			}
			return this;
		}
		
		@RuneAttribute("tradeParty1ReportingDestination")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setTradeParty1ReportingDestination(List<SupervisoryBodyEnum> tradeParty1ReportingDestinations) {
			if (tradeParty1ReportingDestinations == null) {
				this.tradeParty1ReportingDestination = new ArrayList<>();
			} else {
				this.tradeParty1ReportingDestination = tradeParty1ReportingDestinations.stream()
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@RosettaAttribute(value="comment1", isRequired=true)
		@RuneAttribute(value="comment1", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setComment1(String _comment1) {
			this.comment1 = _comment1 == null ? null : _comment1;
			return this;
		}
		
		@RosettaAttribute(value="messageID", isRequired=true)
		@RuneAttribute(value="messageID", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setMessageID(String _messageID) {
			this.messageID = _messageID == null ? null : _messageID;
			return this;
		}
		
		@RosettaAttribute(value="messageType", isRequired=true)
		@RuneAttribute(value="messageType", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setMessageType(String _messageType) {
			this.messageType = _messageType == null ? null : _messageType;
			return this;
		}
		
		@RosettaAttribute(value="version", isRequired=true)
		@RuneAttribute(value="version", isRequired=true)
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder setVersion(String _version) {
			this.version = _version == null ? null : _version;
			return this;
		}
		
		@Override
		public CFTCMarginReport build() {
			return new CFTCMarginReport.CFTCMarginReportImpl(this);
		}
		
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder prune() {
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (getCounterparty1()!=null) return true;
			if (getCounterparty2()!=null) return true;
			if (getCounterparty2IdentifierSource()!=null) return true;
			if (getSubmitterIdentifier()!=null) return true;
			if (getActionType()!=null) return true;
			if (getEventTimestamp()!=null) return true;
			if (getReportingTimestamp()!=null) return true;
			if (getUniqueSwapIdentifier()!=null) return true;
			if (getUniqueTransactionIdentifier()!=null) return true;
			if (getAffiliatedCounterpartyForMarginAndCapitalIndicator()!=null) return true;
			if (getCollateralisationCategory()!=null) return true;
			if (getInitialMarginCollateralPortfolioCode()!=null) return true;
			if (getPortfolioContainingNonReportableComponentIndicator()!=null) return true;
			if (getInitialMarginPostedByTheReportingCounterpartyPostHaircut()!=null) return true;
			if (getInitialMarginPostedByTheReportingCounterpartyPreHaircut()!=null) return true;
			if (getCurrencyOfInitialMarginPosted()!=null) return true;
			if (getInitialMarginCollectedByTheReportingCounterpartyPostHaircut()!=null) return true;
			if (getInitialMarginCollectedByTheReportingCounterpartyPreHaircut()!=null) return true;
			if (getCurrencyOfInitialMarginCollected()!=null) return true;
			if (getVariationMarginCollateralPortfolioCode()!=null) return true;
			if (getVariationMarginPostedByTheReportingCounterpartyPreHaircut()!=null) return true;
			if (getCurrencyOfVariationMarginPosted()!=null) return true;
			if (getVariationMarginCollectedByTheReportingCounterpartyPreHaircut()!=null) return true;
			if (getCurrencyOfVariationMarginCollected()!=null) return true;
			if (getTradeParty1IDType()!=null) return true;
			if (getSubmittingPartyIDType()!=null) return true;
			if (getUsiIDPrefix()!=null) return true;
			if (getUtiIDPrefix()!=null) return true;
			if (getTradeParty1ReportingDestination()!=null && !getTradeParty1ReportingDestination().isEmpty()) return true;
			if (getComment1()!=null) return true;
			if (getMessageID()!=null) return true;
			if (getMessageType()!=null) return true;
			if (getVersion()!=null) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public CFTCMarginReport.CFTCMarginReportBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			CFTCMarginReport.CFTCMarginReportBuilder o = (CFTCMarginReport.CFTCMarginReportBuilder) other;
			
			
			merger.mergeBasic(getCounterparty1(), o.getCounterparty1(), this::setCounterparty1);
			merger.mergeBasic(getCounterparty2(), o.getCounterparty2(), this::setCounterparty2);
			merger.mergeBasic(getCounterparty2IdentifierSource(), o.getCounterparty2IdentifierSource(), this::setCounterparty2IdentifierSource);
			merger.mergeBasic(getSubmitterIdentifier(), o.getSubmitterIdentifier(), this::setSubmitterIdentifier);
			merger.mergeBasic(getActionType(), o.getActionType(), this::setActionType);
			merger.mergeBasic(getEventTimestamp(), o.getEventTimestamp(), this::setEventTimestamp);
			merger.mergeBasic(getReportingTimestamp(), o.getReportingTimestamp(), this::setReportingTimestamp);
			merger.mergeBasic(getUniqueSwapIdentifier(), o.getUniqueSwapIdentifier(), this::setUniqueSwapIdentifier);
			merger.mergeBasic(getUniqueTransactionIdentifier(), o.getUniqueTransactionIdentifier(), this::setUniqueTransactionIdentifier);
			merger.mergeBasic(getAffiliatedCounterpartyForMarginAndCapitalIndicator(), o.getAffiliatedCounterpartyForMarginAndCapitalIndicator(), this::setAffiliatedCounterpartyForMarginAndCapitalIndicator);
			merger.mergeBasic(getCollateralisationCategory(), o.getCollateralisationCategory(), this::setCollateralisationCategory);
			merger.mergeBasic(getInitialMarginCollateralPortfolioCode(), o.getInitialMarginCollateralPortfolioCode(), this::setInitialMarginCollateralPortfolioCode);
			merger.mergeBasic(getPortfolioContainingNonReportableComponentIndicator(), o.getPortfolioContainingNonReportableComponentIndicator(), this::setPortfolioContainingNonReportableComponentIndicator);
			merger.mergeBasic(getInitialMarginPostedByTheReportingCounterpartyPostHaircut(), o.getInitialMarginPostedByTheReportingCounterpartyPostHaircut(), this::setInitialMarginPostedByTheReportingCounterpartyPostHaircut);
			merger.mergeBasic(getInitialMarginPostedByTheReportingCounterpartyPreHaircut(), o.getInitialMarginPostedByTheReportingCounterpartyPreHaircut(), this::setInitialMarginPostedByTheReportingCounterpartyPreHaircut);
			merger.mergeBasic(getCurrencyOfInitialMarginPosted(), o.getCurrencyOfInitialMarginPosted(), this::setCurrencyOfInitialMarginPosted);
			merger.mergeBasic(getInitialMarginCollectedByTheReportingCounterpartyPostHaircut(), o.getInitialMarginCollectedByTheReportingCounterpartyPostHaircut(), this::setInitialMarginCollectedByTheReportingCounterpartyPostHaircut);
			merger.mergeBasic(getInitialMarginCollectedByTheReportingCounterpartyPreHaircut(), o.getInitialMarginCollectedByTheReportingCounterpartyPreHaircut(), this::setInitialMarginCollectedByTheReportingCounterpartyPreHaircut);
			merger.mergeBasic(getCurrencyOfInitialMarginCollected(), o.getCurrencyOfInitialMarginCollected(), this::setCurrencyOfInitialMarginCollected);
			merger.mergeBasic(getVariationMarginCollateralPortfolioCode(), o.getVariationMarginCollateralPortfolioCode(), this::setVariationMarginCollateralPortfolioCode);
			merger.mergeBasic(getVariationMarginPostedByTheReportingCounterpartyPreHaircut(), o.getVariationMarginPostedByTheReportingCounterpartyPreHaircut(), this::setVariationMarginPostedByTheReportingCounterpartyPreHaircut);
			merger.mergeBasic(getCurrencyOfVariationMarginPosted(), o.getCurrencyOfVariationMarginPosted(), this::setCurrencyOfVariationMarginPosted);
			merger.mergeBasic(getVariationMarginCollectedByTheReportingCounterpartyPreHaircut(), o.getVariationMarginCollectedByTheReportingCounterpartyPreHaircut(), this::setVariationMarginCollectedByTheReportingCounterpartyPreHaircut);
			merger.mergeBasic(getCurrencyOfVariationMarginCollected(), o.getCurrencyOfVariationMarginCollected(), this::setCurrencyOfVariationMarginCollected);
			merger.mergeBasic(getTradeParty1IDType(), o.getTradeParty1IDType(), this::setTradeParty1IDType);
			merger.mergeBasic(getSubmittingPartyIDType(), o.getSubmittingPartyIDType(), this::setSubmittingPartyIDType);
			merger.mergeBasic(getUsiIDPrefix(), o.getUsiIDPrefix(), this::setUsiIDPrefix);
			merger.mergeBasic(getUtiIDPrefix(), o.getUtiIDPrefix(), this::setUtiIDPrefix);
			merger.mergeBasic(getTradeParty1ReportingDestination(), o.getTradeParty1ReportingDestination(), (Consumer<SupervisoryBodyEnum>) this::addTradeParty1ReportingDestination);
			merger.mergeBasic(getComment1(), o.getComment1(), this::setComment1);
			merger.mergeBasic(getMessageID(), o.getMessageID(), this::setMessageID);
			merger.mergeBasic(getMessageType(), o.getMessageType(), this::setMessageType);
			merger.mergeBasic(getVersion(), o.getVersion(), this::setVersion);
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			CFTCMarginReport _that = getType().cast(o);
		
			if (!Objects.equals(counterparty1, _that.getCounterparty1())) return false;
			if (!Objects.equals(counterparty2, _that.getCounterparty2())) return false;
			if (!Objects.equals(counterparty2IdentifierSource, _that.getCounterparty2IdentifierSource())) return false;
			if (!Objects.equals(submitterIdentifier, _that.getSubmitterIdentifier())) return false;
			if (!Objects.equals(actionType, _that.getActionType())) return false;
			if (!Objects.equals(eventTimestamp, _that.getEventTimestamp())) return false;
			if (!Objects.equals(reportingTimestamp, _that.getReportingTimestamp())) return false;
			if (!Objects.equals(uniqueSwapIdentifier, _that.getUniqueSwapIdentifier())) return false;
			if (!Objects.equals(uniqueTransactionIdentifier, _that.getUniqueTransactionIdentifier())) return false;
			if (!Objects.equals(affiliatedCounterpartyForMarginAndCapitalIndicator, _that.getAffiliatedCounterpartyForMarginAndCapitalIndicator())) return false;
			if (!Objects.equals(collateralisationCategory, _that.getCollateralisationCategory())) return false;
			if (!Objects.equals(initialMarginCollateralPortfolioCode, _that.getInitialMarginCollateralPortfolioCode())) return false;
			if (!Objects.equals(portfolioContainingNonReportableComponentIndicator, _that.getPortfolioContainingNonReportableComponentIndicator())) return false;
			if (!Objects.equals(initialMarginPostedByTheReportingCounterpartyPostHaircut, _that.getInitialMarginPostedByTheReportingCounterpartyPostHaircut())) return false;
			if (!Objects.equals(initialMarginPostedByTheReportingCounterpartyPreHaircut, _that.getInitialMarginPostedByTheReportingCounterpartyPreHaircut())) return false;
			if (!Objects.equals(currencyOfInitialMarginPosted, _that.getCurrencyOfInitialMarginPosted())) return false;
			if (!Objects.equals(initialMarginCollectedByTheReportingCounterpartyPostHaircut, _that.getInitialMarginCollectedByTheReportingCounterpartyPostHaircut())) return false;
			if (!Objects.equals(initialMarginCollectedByTheReportingCounterpartyPreHaircut, _that.getInitialMarginCollectedByTheReportingCounterpartyPreHaircut())) return false;
			if (!Objects.equals(currencyOfInitialMarginCollected, _that.getCurrencyOfInitialMarginCollected())) return false;
			if (!Objects.equals(variationMarginCollateralPortfolioCode, _that.getVariationMarginCollateralPortfolioCode())) return false;
			if (!Objects.equals(variationMarginPostedByTheReportingCounterpartyPreHaircut, _that.getVariationMarginPostedByTheReportingCounterpartyPreHaircut())) return false;
			if (!Objects.equals(currencyOfVariationMarginPosted, _that.getCurrencyOfVariationMarginPosted())) return false;
			if (!Objects.equals(variationMarginCollectedByTheReportingCounterpartyPreHaircut, _that.getVariationMarginCollectedByTheReportingCounterpartyPreHaircut())) return false;
			if (!Objects.equals(currencyOfVariationMarginCollected, _that.getCurrencyOfVariationMarginCollected())) return false;
			if (!Objects.equals(tradeParty1IDType, _that.getTradeParty1IDType())) return false;
			if (!Objects.equals(submittingPartyIDType, _that.getSubmittingPartyIDType())) return false;
			if (!Objects.equals(usiIDPrefix, _that.getUsiIDPrefix())) return false;
			if (!Objects.equals(utiIDPrefix, _that.getUtiIDPrefix())) return false;
			if (!ListEquals.listEquals(tradeParty1ReportingDestination, _that.getTradeParty1ReportingDestination())) return false;
			if (!Objects.equals(comment1, _that.getComment1())) return false;
			if (!Objects.equals(messageID, _that.getMessageID())) return false;
			if (!Objects.equals(messageType, _that.getMessageType())) return false;
			if (!Objects.equals(version, _that.getVersion())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (counterparty1 != null ? counterparty1.hashCode() : 0);
			_result = 31 * _result + (counterparty2 != null ? counterparty2.hashCode() : 0);
			_result = 31 * _result + (counterparty2IdentifierSource != null ? counterparty2IdentifierSource.hashCode() : 0);
			_result = 31 * _result + (submitterIdentifier != null ? submitterIdentifier.hashCode() : 0);
			_result = 31 * _result + (actionType != null ? actionType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (eventTimestamp != null ? eventTimestamp.hashCode() : 0);
			_result = 31 * _result + (reportingTimestamp != null ? reportingTimestamp.hashCode() : 0);
			_result = 31 * _result + (uniqueSwapIdentifier != null ? uniqueSwapIdentifier.hashCode() : 0);
			_result = 31 * _result + (uniqueTransactionIdentifier != null ? uniqueTransactionIdentifier.hashCode() : 0);
			_result = 31 * _result + (affiliatedCounterpartyForMarginAndCapitalIndicator != null ? affiliatedCounterpartyForMarginAndCapitalIndicator.hashCode() : 0);
			_result = 31 * _result + (collateralisationCategory != null ? collateralisationCategory.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (initialMarginCollateralPortfolioCode != null ? initialMarginCollateralPortfolioCode.hashCode() : 0);
			_result = 31 * _result + (portfolioContainingNonReportableComponentIndicator != null ? portfolioContainingNonReportableComponentIndicator.hashCode() : 0);
			_result = 31 * _result + (initialMarginPostedByTheReportingCounterpartyPostHaircut != null ? initialMarginPostedByTheReportingCounterpartyPostHaircut.hashCode() : 0);
			_result = 31 * _result + (initialMarginPostedByTheReportingCounterpartyPreHaircut != null ? initialMarginPostedByTheReportingCounterpartyPreHaircut.hashCode() : 0);
			_result = 31 * _result + (currencyOfInitialMarginPosted != null ? currencyOfInitialMarginPosted.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (initialMarginCollectedByTheReportingCounterpartyPostHaircut != null ? initialMarginCollectedByTheReportingCounterpartyPostHaircut.hashCode() : 0);
			_result = 31 * _result + (initialMarginCollectedByTheReportingCounterpartyPreHaircut != null ? initialMarginCollectedByTheReportingCounterpartyPreHaircut.hashCode() : 0);
			_result = 31 * _result + (currencyOfInitialMarginCollected != null ? currencyOfInitialMarginCollected.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (variationMarginCollateralPortfolioCode != null ? variationMarginCollateralPortfolioCode.hashCode() : 0);
			_result = 31 * _result + (variationMarginPostedByTheReportingCounterpartyPreHaircut != null ? variationMarginPostedByTheReportingCounterpartyPreHaircut.hashCode() : 0);
			_result = 31 * _result + (currencyOfVariationMarginPosted != null ? currencyOfVariationMarginPosted.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (variationMarginCollectedByTheReportingCounterpartyPreHaircut != null ? variationMarginCollectedByTheReportingCounterpartyPreHaircut.hashCode() : 0);
			_result = 31 * _result + (currencyOfVariationMarginCollected != null ? currencyOfVariationMarginCollected.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (tradeParty1IDType != null ? tradeParty1IDType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (submittingPartyIDType != null ? submittingPartyIDType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (usiIDPrefix != null ? usiIDPrefix.hashCode() : 0);
			_result = 31 * _result + (utiIDPrefix != null ? utiIDPrefix.hashCode() : 0);
			_result = 31 * _result + (tradeParty1ReportingDestination != null ? tradeParty1ReportingDestination.stream().map(Object::getClass).map(Class::getName).mapToInt(String::hashCode).sum() : 0);
			_result = 31 * _result + (comment1 != null ? comment1.hashCode() : 0);
			_result = 31 * _result + (messageID != null ? messageID.hashCode() : 0);
			_result = 31 * _result + (messageType != null ? messageType.hashCode() : 0);
			_result = 31 * _result + (version != null ? version.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "CFTCMarginReportBuilder {" +
				"counterparty1=" + this.counterparty1 + ", " +
				"counterparty2=" + this.counterparty2 + ", " +
				"counterparty2IdentifierSource=" + this.counterparty2IdentifierSource + ", " +
				"submitterIdentifier=" + this.submitterIdentifier + ", " +
				"actionType=" + this.actionType + ", " +
				"eventTimestamp=" + this.eventTimestamp + ", " +
				"reportingTimestamp=" + this.reportingTimestamp + ", " +
				"uniqueSwapIdentifier=" + this.uniqueSwapIdentifier + ", " +
				"uniqueTransactionIdentifier=" + this.uniqueTransactionIdentifier + ", " +
				"affiliatedCounterpartyForMarginAndCapitalIndicator=" + this.affiliatedCounterpartyForMarginAndCapitalIndicator + ", " +
				"collateralisationCategory=" + this.collateralisationCategory + ", " +
				"initialMarginCollateralPortfolioCode=" + this.initialMarginCollateralPortfolioCode + ", " +
				"portfolioContainingNonReportableComponentIndicator=" + this.portfolioContainingNonReportableComponentIndicator + ", " +
				"initialMarginPostedByTheReportingCounterpartyPostHaircut=" + this.initialMarginPostedByTheReportingCounterpartyPostHaircut + ", " +
				"initialMarginPostedByTheReportingCounterpartyPreHaircut=" + this.initialMarginPostedByTheReportingCounterpartyPreHaircut + ", " +
				"currencyOfInitialMarginPosted=" + this.currencyOfInitialMarginPosted + ", " +
				"initialMarginCollectedByTheReportingCounterpartyPostHaircut=" + this.initialMarginCollectedByTheReportingCounterpartyPostHaircut + ", " +
				"initialMarginCollectedByTheReportingCounterpartyPreHaircut=" + this.initialMarginCollectedByTheReportingCounterpartyPreHaircut + ", " +
				"currencyOfInitialMarginCollected=" + this.currencyOfInitialMarginCollected + ", " +
				"variationMarginCollateralPortfolioCode=" + this.variationMarginCollateralPortfolioCode + ", " +
				"variationMarginPostedByTheReportingCounterpartyPreHaircut=" + this.variationMarginPostedByTheReportingCounterpartyPreHaircut + ", " +
				"currencyOfVariationMarginPosted=" + this.currencyOfVariationMarginPosted + ", " +
				"variationMarginCollectedByTheReportingCounterpartyPreHaircut=" + this.variationMarginCollectedByTheReportingCounterpartyPreHaircut + ", " +
				"currencyOfVariationMarginCollected=" + this.currencyOfVariationMarginCollected + ", " +
				"tradeParty1IDType=" + this.tradeParty1IDType + ", " +
				"submittingPartyIDType=" + this.submittingPartyIDType + ", " +
				"usiIDPrefix=" + this.usiIDPrefix + ", " +
				"utiIDPrefix=" + this.utiIDPrefix + ", " +
				"tradeParty1ReportingDestination=" + this.tradeParty1ReportingDestination + ", " +
				"comment1=" + this.comment1 + ", " +
				"messageID=" + this.messageID + ", " +
				"messageType=" + this.messageType + ", " +
				"version=" + this.version +
			'}';
		}
	}
}
