package drr.regulation.common;

import cdm.event.common.PositionIdentifier;
import cdm.observable.asset.PriceSchedule;
import com.google.common.collect.ImmutableList;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import com.rosetta.model.lib.records.Date;
import com.rosetta.util.ListEquals;
import drr.base.qualification.event.ReportableActionEnum;
import drr.base.trade.ConfirmationMethodEnum;
import drr.base.trade.ExecutionVenueTypeEnum;
import drr.base.trade.ReportableInformationBase;
import drr.base.trade.ReportablePriceSource;
import drr.base.trade.basket.CustomBasket;
import drr.enrichment.common.EnrichmentData;
import drr.regulation.common.meta.ReportableInformationMeta;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static java.util.Optional.ofNullable;

/**
 * Specifies the relevant regulators or other supervisory bodies the transaction is relevant for, and regime specific information required relating to the transaction and the parties.
 * @version 7.0.0-dev.73
 */
@RosettaDataType(value="ReportableInformation", builder=ReportableInformation.ReportableInformationBuilderImpl.class, version="7.0.0-dev.73")
@RuneDataType(value="ReportableInformation", model="drr", builder=ReportableInformation.ReportableInformationBuilderImpl.class, version="7.0.0-dev.73")
public interface ReportableInformation extends ReportableInformationBase {

	ReportableInformationMeta metaData = new ReportableInformationMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 * Specifies party specific information required for reporting of the transaction
	 */
	List<? extends PartyInformation> getPartyInformation();
	List<? extends TransactionInformation> getTransactionInformation();

	/*********************** Build Methods  ***********************/
	ReportableInformation build();
	
	ReportableInformation.ReportableInformationBuilder toBuilder();
	
	static ReportableInformation.ReportableInformationBuilder builder() {
		return new ReportableInformation.ReportableInformationBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends ReportableInformation> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends ReportableInformation> getType() {
		return ReportableInformation.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processor.processBasic(path.newSubPath("confirmationMethod"), ConfirmationMethodEnum.class, getConfirmationMethod(), this);
		processor.processBasic(path.newSubPath("executionVenueType"), ExecutionVenueTypeEnum.class, getExecutionVenueType(), this);
		processor.processBasic(path.newSubPath("intragroup"), Boolean.class, getIntragroup(), this);
		processor.processBasic(path.newSubPath("affiliated"), Boolean.class, getAffiliated(), this);
		processor.processBasic(path.newSubPath("largeSizeTrade"), Boolean.class, getLargeSizeTrade(), this);
		processor.processBasic(path.newSubPath("sefOrDcmAnonymousIndicator"), Boolean.class, getSefOrDcmAnonymousIndicator(), this);
		processor.processBasic(path.newSubPath("originalExecutionTimestamp"), ZonedDateTime.class, getOriginalExecutionTimestamp(), this);
		processor.processBasic(path.newSubPath("cryptoBased"), Boolean.class, getCryptoBased(), this);
		processRosetta(path.newSubPath("underlyingAssetLastAvailableSpotPrice"), processor, PriceSchedule.class, getUnderlyingAssetLastAvailableSpotPrice());
		processRosetta(path.newSubPath("underlyingAssetPriceSource"), processor, ReportablePriceSource.class, getUnderlyingAssetPriceSource());
		processor.processBasic(path.newSubPath("reportableAction"), ReportableActionEnum.class, getReportableAction(), this);
		processRosetta(path.newSubPath("subsequentPositionUTI"), processor, PositionIdentifier.class, getSubsequentPositionUTI());
		processRosetta(path.newSubPath("counterpartyPositionPriorUTI"), processor, PositionIdentifier.class, getCounterpartyPositionPriorUTI());
		processRosetta(path.newSubPath("customBasket"), processor, CustomBasket.class, getCustomBasket());
		processor.processBasic(path.newSubPath("finalContractualSettlementDate"), Date.class, getFinalContractualSettlementDate(), this);
		processRosetta(path.newSubPath("enrichment"), processor, EnrichmentData.class, getEnrichment());
		processRosetta(path.newSubPath("partyInformation"), processor, PartyInformation.class, getPartyInformation());
		processRosetta(path.newSubPath("transactionInformation"), processor, TransactionInformation.class, getTransactionInformation());
	}
	

	/*********************** Builder Interface  ***********************/
	interface ReportableInformationBuilder extends ReportableInformation, ReportableInformationBase.ReportableInformationBaseBuilder {
		PartyInformation.PartyInformationBuilder getOrCreatePartyInformation(int index);
		@Override
		List<? extends PartyInformation.PartyInformationBuilder> getPartyInformation();
		TransactionInformation.TransactionInformationBuilder getOrCreateTransactionInformation(int index);
		@Override
		List<? extends TransactionInformation.TransactionInformationBuilder> getTransactionInformation();
		@Override
		ReportableInformation.ReportableInformationBuilder setConfirmationMethod(ConfirmationMethodEnum confirmationMethod);
		@Override
		ReportableInformation.ReportableInformationBuilder setExecutionVenueType(ExecutionVenueTypeEnum executionVenueType);
		@Override
		ReportableInformation.ReportableInformationBuilder setIntragroup(Boolean intragroup);
		@Override
		ReportableInformation.ReportableInformationBuilder setAffiliated(Boolean affiliated);
		@Override
		ReportableInformation.ReportableInformationBuilder setLargeSizeTrade(Boolean largeSizeTrade);
		@Override
		ReportableInformation.ReportableInformationBuilder setSefOrDcmAnonymousIndicator(Boolean sefOrDcmAnonymousIndicator);
		@Override
		ReportableInformation.ReportableInformationBuilder setOriginalExecutionTimestamp(ZonedDateTime originalExecutionTimestamp);
		@Override
		ReportableInformation.ReportableInformationBuilder setCryptoBased(Boolean cryptoBased);
		@Override
		ReportableInformation.ReportableInformationBuilder setUnderlyingAssetLastAvailableSpotPrice(PriceSchedule underlyingAssetLastAvailableSpotPrice);
		@Override
		ReportableInformation.ReportableInformationBuilder addUnderlyingAssetPriceSource(ReportablePriceSource underlyingAssetPriceSource);
		@Override
		ReportableInformation.ReportableInformationBuilder addUnderlyingAssetPriceSource(ReportablePriceSource underlyingAssetPriceSource, int idx);
		@Override
		ReportableInformation.ReportableInformationBuilder addUnderlyingAssetPriceSource(List<? extends ReportablePriceSource> underlyingAssetPriceSource);
		@Override
		ReportableInformation.ReportableInformationBuilder setUnderlyingAssetPriceSource(List<? extends ReportablePriceSource> underlyingAssetPriceSource);
		@Override
		ReportableInformation.ReportableInformationBuilder setReportableAction(ReportableActionEnum reportableAction);
		@Override
		ReportableInformation.ReportableInformationBuilder setSubsequentPositionUTI(PositionIdentifier subsequentPositionUTI);
		@Override
		ReportableInformation.ReportableInformationBuilder setCounterpartyPositionPriorUTI(PositionIdentifier counterpartyPositionPriorUTI);
		@Override
		ReportableInformation.ReportableInformationBuilder setCustomBasket(CustomBasket customBasket);
		@Override
		ReportableInformation.ReportableInformationBuilder setFinalContractualSettlementDate(Date finalContractualSettlementDate);
		@Override
		ReportableInformation.ReportableInformationBuilder setEnrichment(EnrichmentData enrichment);
		ReportableInformation.ReportableInformationBuilder addPartyInformation(PartyInformation partyInformation);
		ReportableInformation.ReportableInformationBuilder addPartyInformation(PartyInformation partyInformation, int idx);
		ReportableInformation.ReportableInformationBuilder addPartyInformation(List<? extends PartyInformation> partyInformation);
		ReportableInformation.ReportableInformationBuilder setPartyInformation(List<? extends PartyInformation> partyInformation);
		ReportableInformation.ReportableInformationBuilder addTransactionInformation(TransactionInformation transactionInformation);
		ReportableInformation.ReportableInformationBuilder addTransactionInformation(TransactionInformation transactionInformation, int idx);
		ReportableInformation.ReportableInformationBuilder addTransactionInformation(List<? extends TransactionInformation> transactionInformation);
		ReportableInformation.ReportableInformationBuilder setTransactionInformation(List<? extends TransactionInformation> transactionInformation);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processor.processBasic(path.newSubPath("confirmationMethod"), ConfirmationMethodEnum.class, getConfirmationMethod(), this);
			processor.processBasic(path.newSubPath("executionVenueType"), ExecutionVenueTypeEnum.class, getExecutionVenueType(), this);
			processor.processBasic(path.newSubPath("intragroup"), Boolean.class, getIntragroup(), this);
			processor.processBasic(path.newSubPath("affiliated"), Boolean.class, getAffiliated(), this);
			processor.processBasic(path.newSubPath("largeSizeTrade"), Boolean.class, getLargeSizeTrade(), this);
			processor.processBasic(path.newSubPath("sefOrDcmAnonymousIndicator"), Boolean.class, getSefOrDcmAnonymousIndicator(), this);
			processor.processBasic(path.newSubPath("originalExecutionTimestamp"), ZonedDateTime.class, getOriginalExecutionTimestamp(), this);
			processor.processBasic(path.newSubPath("cryptoBased"), Boolean.class, getCryptoBased(), this);
			processRosetta(path.newSubPath("underlyingAssetLastAvailableSpotPrice"), processor, PriceSchedule.PriceScheduleBuilder.class, getUnderlyingAssetLastAvailableSpotPrice());
			processRosetta(path.newSubPath("underlyingAssetPriceSource"), processor, ReportablePriceSource.ReportablePriceSourceBuilder.class, getUnderlyingAssetPriceSource());
			processor.processBasic(path.newSubPath("reportableAction"), ReportableActionEnum.class, getReportableAction(), this);
			processRosetta(path.newSubPath("subsequentPositionUTI"), processor, PositionIdentifier.PositionIdentifierBuilder.class, getSubsequentPositionUTI());
			processRosetta(path.newSubPath("counterpartyPositionPriorUTI"), processor, PositionIdentifier.PositionIdentifierBuilder.class, getCounterpartyPositionPriorUTI());
			processRosetta(path.newSubPath("customBasket"), processor, CustomBasket.CustomBasketBuilder.class, getCustomBasket());
			processor.processBasic(path.newSubPath("finalContractualSettlementDate"), Date.class, getFinalContractualSettlementDate(), this);
			processRosetta(path.newSubPath("enrichment"), processor, EnrichmentData.EnrichmentDataBuilder.class, getEnrichment());
			processRosetta(path.newSubPath("partyInformation"), processor, PartyInformation.PartyInformationBuilder.class, getPartyInformation());
			processRosetta(path.newSubPath("transactionInformation"), processor, TransactionInformation.TransactionInformationBuilder.class, getTransactionInformation());
		}
		

		ReportableInformation.ReportableInformationBuilder prune();
	}

	/*********************** Immutable Implementation of ReportableInformation  ***********************/
	class ReportableInformationImpl extends ReportableInformationBase.ReportableInformationBaseImpl implements ReportableInformation {
		private final List<? extends PartyInformation> partyInformation;
		private final List<? extends TransactionInformation> transactionInformation;
		
		protected ReportableInformationImpl(ReportableInformation.ReportableInformationBuilder builder) {
			super(builder);
			this.partyInformation = ofNullable(builder.getPartyInformation()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.transactionInformation = ofNullable(builder.getTransactionInformation()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
		}
		
		@Override
		@RosettaAttribute(value="partyInformation", isRequired=true)
		@RuneAttribute(value="partyInformation", isRequired=true)
		public List<? extends PartyInformation> getPartyInformation() {
			return partyInformation;
		}
		
		@Override
		@RosettaAttribute("transactionInformation")
		@RuneAttribute("transactionInformation")
		public List<? extends TransactionInformation> getTransactionInformation() {
			return transactionInformation;
		}
		
		@Override
		public ReportableInformation build() {
			return this;
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder toBuilder() {
			ReportableInformation.ReportableInformationBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(ReportableInformation.ReportableInformationBuilder builder) {
			super.setBuilderFields(builder);
			ofNullable(getPartyInformation()).ifPresent(builder::setPartyInformation);
			ofNullable(getTransactionInformation()).ifPresent(builder::setTransactionInformation);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			ReportableInformation _that = getType().cast(o);
		
			if (!ListEquals.listEquals(partyInformation, _that.getPartyInformation())) return false;
			if (!ListEquals.listEquals(transactionInformation, _that.getTransactionInformation())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (partyInformation != null ? partyInformation.hashCode() : 0);
			_result = 31 * _result + (transactionInformation != null ? transactionInformation.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ReportableInformation {" +
				"partyInformation=" + this.partyInformation + ", " +
				"transactionInformation=" + this.transactionInformation +
			'}' + " " + super.toString();
		}
	}

	/*********************** Builder Implementation of ReportableInformation  ***********************/
	class ReportableInformationBuilderImpl extends ReportableInformationBase.ReportableInformationBaseBuilderImpl implements ReportableInformation.ReportableInformationBuilder {
	
		protected List<PartyInformation.PartyInformationBuilder> partyInformation = new ArrayList<>();
		protected List<TransactionInformation.TransactionInformationBuilder> transactionInformation = new ArrayList<>();
		
		@Override
		@RosettaAttribute(value="partyInformation", isRequired=true)
		@RuneAttribute(value="partyInformation", isRequired=true)
		public List<? extends PartyInformation.PartyInformationBuilder> getPartyInformation() {
			return partyInformation;
		}
		
		@Override
		public PartyInformation.PartyInformationBuilder getOrCreatePartyInformation(int index) {
			if (partyInformation==null) {
				this.partyInformation = new ArrayList<>();
			}
			return getIndex(partyInformation, index, () -> {
						PartyInformation.PartyInformationBuilder newPartyInformation = PartyInformation.builder();
						return newPartyInformation;
					});
		}
		
		@Override
		@RosettaAttribute("transactionInformation")
		@RuneAttribute("transactionInformation")
		public List<? extends TransactionInformation.TransactionInformationBuilder> getTransactionInformation() {
			return transactionInformation;
		}
		
		@Override
		public TransactionInformation.TransactionInformationBuilder getOrCreateTransactionInformation(int index) {
			if (transactionInformation==null) {
				this.transactionInformation = new ArrayList<>();
			}
			return getIndex(transactionInformation, index, () -> {
						TransactionInformation.TransactionInformationBuilder newTransactionInformation = TransactionInformation.builder();
						return newTransactionInformation;
					});
		}
		
		@RosettaAttribute("confirmationMethod")
		@RuneAttribute("confirmationMethod")
		@Override
		public ReportableInformation.ReportableInformationBuilder setConfirmationMethod(ConfirmationMethodEnum _confirmationMethod) {
			this.confirmationMethod = _confirmationMethod == null ? null : _confirmationMethod;
			return this;
		}
		
		@RosettaAttribute(value="executionVenueType", isRequired=true)
		@RuneAttribute(value="executionVenueType", isRequired=true)
		@Override
		public ReportableInformation.ReportableInformationBuilder setExecutionVenueType(ExecutionVenueTypeEnum _executionVenueType) {
			this.executionVenueType = _executionVenueType == null ? null : _executionVenueType;
			return this;
		}
		
		@RosettaAttribute("intragroup")
		@RuneAttribute("intragroup")
		@Override
		public ReportableInformation.ReportableInformationBuilder setIntragroup(Boolean _intragroup) {
			this.intragroup = _intragroup == null ? null : _intragroup;
			return this;
		}
		
		@RosettaAttribute("affiliated")
		@RuneAttribute("affiliated")
		@Override
		public ReportableInformation.ReportableInformationBuilder setAffiliated(Boolean _affiliated) {
			this.affiliated = _affiliated == null ? null : _affiliated;
			return this;
		}
		
		@RosettaAttribute("largeSizeTrade")
		@RuneAttribute("largeSizeTrade")
		@Override
		public ReportableInformation.ReportableInformationBuilder setLargeSizeTrade(Boolean _largeSizeTrade) {
			this.largeSizeTrade = _largeSizeTrade == null ? null : _largeSizeTrade;
			return this;
		}
		
		@RosettaAttribute("sefOrDcmAnonymousIndicator")
		@RuneAttribute("sefOrDcmAnonymousIndicator")
		@Override
		public ReportableInformation.ReportableInformationBuilder setSefOrDcmAnonymousIndicator(Boolean _sefOrDcmAnonymousIndicator) {
			this.sefOrDcmAnonymousIndicator = _sefOrDcmAnonymousIndicator == null ? null : _sefOrDcmAnonymousIndicator;
			return this;
		}
		
		@RosettaAttribute("originalExecutionTimestamp")
		@RuneAttribute("originalExecutionTimestamp")
		@Override
		public ReportableInformation.ReportableInformationBuilder setOriginalExecutionTimestamp(ZonedDateTime _originalExecutionTimestamp) {
			this.originalExecutionTimestamp = _originalExecutionTimestamp == null ? null : _originalExecutionTimestamp;
			return this;
		}
		
		@RosettaAttribute("cryptoBased")
		@RuneAttribute("cryptoBased")
		@Override
		public ReportableInformation.ReportableInformationBuilder setCryptoBased(Boolean _cryptoBased) {
			this.cryptoBased = _cryptoBased == null ? null : _cryptoBased;
			return this;
		}
		
		@RosettaAttribute("underlyingAssetLastAvailableSpotPrice")
		@RuneAttribute("underlyingAssetLastAvailableSpotPrice")
		@Override
		public ReportableInformation.ReportableInformationBuilder setUnderlyingAssetLastAvailableSpotPrice(PriceSchedule _underlyingAssetLastAvailableSpotPrice) {
			this.underlyingAssetLastAvailableSpotPrice = _underlyingAssetLastAvailableSpotPrice == null ? null : _underlyingAssetLastAvailableSpotPrice.toBuilder();
			return this;
		}
		
		@RosettaAttribute("underlyingAssetPriceSource")
		@RuneAttribute("underlyingAssetPriceSource")
		@Override
		public ReportableInformation.ReportableInformationBuilder addUnderlyingAssetPriceSource(ReportablePriceSource _underlyingAssetPriceSource) {
			if (_underlyingAssetPriceSource != null) {
				this.underlyingAssetPriceSource.add(_underlyingAssetPriceSource.toBuilder());
			}
			return this;
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder addUnderlyingAssetPriceSource(ReportablePriceSource _underlyingAssetPriceSource, int idx) {
			getIndex(this.underlyingAssetPriceSource, idx, () -> _underlyingAssetPriceSource.toBuilder());
			return this;
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder addUnderlyingAssetPriceSource(List<? extends ReportablePriceSource> underlyingAssetPriceSources) {
			if (underlyingAssetPriceSources != null) {
				for (final ReportablePriceSource toAdd : underlyingAssetPriceSources) {
					this.underlyingAssetPriceSource.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@RuneAttribute("underlyingAssetPriceSource")
		@Override
		public ReportableInformation.ReportableInformationBuilder setUnderlyingAssetPriceSource(List<? extends ReportablePriceSource> underlyingAssetPriceSources) {
			if (underlyingAssetPriceSources == null) {
				this.underlyingAssetPriceSource = new ArrayList<>();
			} else {
				this.underlyingAssetPriceSource = underlyingAssetPriceSources.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@RosettaAttribute("reportableAction")
		@RuneAttribute("reportableAction")
		@Override
		public ReportableInformation.ReportableInformationBuilder setReportableAction(ReportableActionEnum _reportableAction) {
			this.reportableAction = _reportableAction == null ? null : _reportableAction;
			return this;
		}
		
		@RosettaAttribute("subsequentPositionUTI")
		@RuneAttribute("subsequentPositionUTI")
		@Override
		public ReportableInformation.ReportableInformationBuilder setSubsequentPositionUTI(PositionIdentifier _subsequentPositionUTI) {
			this.subsequentPositionUTI = _subsequentPositionUTI == null ? null : _subsequentPositionUTI.toBuilder();
			return this;
		}
		
		@RosettaAttribute("counterpartyPositionPriorUTI")
		@RuneAttribute("counterpartyPositionPriorUTI")
		@Override
		public ReportableInformation.ReportableInformationBuilder setCounterpartyPositionPriorUTI(PositionIdentifier _counterpartyPositionPriorUTI) {
			this.counterpartyPositionPriorUTI = _counterpartyPositionPriorUTI == null ? null : _counterpartyPositionPriorUTI.toBuilder();
			return this;
		}
		
		@RosettaAttribute("customBasket")
		@RuneAttribute("customBasket")
		@Override
		public ReportableInformation.ReportableInformationBuilder setCustomBasket(CustomBasket _customBasket) {
			this.customBasket = _customBasket == null ? null : _customBasket.toBuilder();
			return this;
		}
		
		@RosettaAttribute("finalContractualSettlementDate")
		@RuneAttribute("finalContractualSettlementDate")
		@Override
		public ReportableInformation.ReportableInformationBuilder setFinalContractualSettlementDate(Date _finalContractualSettlementDate) {
			this.finalContractualSettlementDate = _finalContractualSettlementDate == null ? null : _finalContractualSettlementDate;
			return this;
		}
		
		@RosettaAttribute("enrichment")
		@RuneAttribute("enrichment")
		@Override
		public ReportableInformation.ReportableInformationBuilder setEnrichment(EnrichmentData _enrichment) {
			this.enrichment = _enrichment == null ? null : _enrichment.toBuilder();
			return this;
		}
		
		@RosettaAttribute(value="partyInformation", isRequired=true)
		@RuneAttribute(value="partyInformation", isRequired=true)
		@Override
		public ReportableInformation.ReportableInformationBuilder addPartyInformation(PartyInformation _partyInformation) {
			if (_partyInformation != null) {
				this.partyInformation.add(_partyInformation.toBuilder());
			}
			return this;
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder addPartyInformation(PartyInformation _partyInformation, int idx) {
			getIndex(this.partyInformation, idx, () -> _partyInformation.toBuilder());
			return this;
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder addPartyInformation(List<? extends PartyInformation> partyInformations) {
			if (partyInformations != null) {
				for (final PartyInformation toAdd : partyInformations) {
					this.partyInformation.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@RuneAttribute("partyInformation")
		@Override
		public ReportableInformation.ReportableInformationBuilder setPartyInformation(List<? extends PartyInformation> partyInformations) {
			if (partyInformations == null) {
				this.partyInformation = new ArrayList<>();
			} else {
				this.partyInformation = partyInformations.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@RosettaAttribute("transactionInformation")
		@RuneAttribute("transactionInformation")
		@Override
		public ReportableInformation.ReportableInformationBuilder addTransactionInformation(TransactionInformation _transactionInformation) {
			if (_transactionInformation != null) {
				this.transactionInformation.add(_transactionInformation.toBuilder());
			}
			return this;
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder addTransactionInformation(TransactionInformation _transactionInformation, int idx) {
			getIndex(this.transactionInformation, idx, () -> _transactionInformation.toBuilder());
			return this;
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder addTransactionInformation(List<? extends TransactionInformation> transactionInformations) {
			if (transactionInformations != null) {
				for (final TransactionInformation toAdd : transactionInformations) {
					this.transactionInformation.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@RuneAttribute("transactionInformation")
		@Override
		public ReportableInformation.ReportableInformationBuilder setTransactionInformation(List<? extends TransactionInformation> transactionInformations) {
			if (transactionInformations == null) {
				this.transactionInformation = new ArrayList<>();
			} else {
				this.transactionInformation = transactionInformations.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		public ReportableInformation build() {
			return new ReportableInformation.ReportableInformationImpl(this);
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ReportableInformation.ReportableInformationBuilder prune() {
			super.prune();
			partyInformation = partyInformation.stream().filter(b->b!=null).<PartyInformation.PartyInformationBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			transactionInformation = transactionInformation.stream().filter(b->b!=null).<TransactionInformation.TransactionInformationBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (super.hasData()) return true;
			if (getPartyInformation()!=null && getPartyInformation().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getTransactionInformation()!=null && getTransactionInformation().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ReportableInformation.ReportableInformationBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			super.merge(other, merger);
			ReportableInformation.ReportableInformationBuilder o = (ReportableInformation.ReportableInformationBuilder) other;
			
			merger.mergeRosetta(getPartyInformation(), o.getPartyInformation(), this::getOrCreatePartyInformation);
			merger.mergeRosetta(getTransactionInformation(), o.getTransactionInformation(), this::getOrCreateTransactionInformation);
			
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			ReportableInformation _that = getType().cast(o);
		
			if (!ListEquals.listEquals(partyInformation, _that.getPartyInformation())) return false;
			if (!ListEquals.listEquals(transactionInformation, _that.getTransactionInformation())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (partyInformation != null ? partyInformation.hashCode() : 0);
			_result = 31 * _result + (transactionInformation != null ? transactionInformation.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ReportableInformationBuilder {" +
				"partyInformation=" + this.partyInformation + ", " +
				"transactionInformation=" + this.transactionInformation +
			'}' + " " + super.toString();
		}
	}
}
