package drr.regulation.jfsa.rewrite.valuation;

import cdm.base.staticdata.asset.common.ISOCurrencyCodeEnum;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RosettaIgnore;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.annotations.RuneIgnore;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import com.rosetta.model.lib.records.Date;
import drr.regulation.common.valuation.CommonValuationReport;
import drr.regulation.jfsa.rewrite.valuation.meta.JFSAValuationReportMeta;
import drr.standards.iso.ActionTypeEnum;
import drr.standards.iso.ReportLevelEnum;
import drr.standards.iso.ValuationType1Code;
import java.math.BigDecimal;
import java.time.ZonedDateTime;
import java.util.Objects;

import static java.util.Optional.ofNullable;

/**
 * @version 7.0.0-dev.92
 */
@RosettaDataType(value="JFSAValuationReport", builder=JFSAValuationReport.JFSAValuationReportBuilderImpl.class, version="7.0.0-dev.92")
@RuneDataType(value="JFSAValuationReport", model="drr", builder=JFSAValuationReport.JFSAValuationReportBuilderImpl.class, version="7.0.0-dev.92")
public interface JFSAValuationReport extends CommonValuationReport {

	JFSAValuationReportMeta metaData = new JFSAValuationReportMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "6" * field "Entity Responsible for Reporting"
	 *
	 * Provision The identifier of the financial institution that is the reporting obligation.
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "6" * field "Entity Responsible for Reporting - Remark"
	 *
	 * Provision Reporting party of the trades will be determined based on the identifier reported for 'Entity responsible for reporting' and by 'Submitter identifier'. Examples of who the reporting party is:
	             [Case 1] Where a party to a transaction is other than a fund (normal case):
	             'Entity responsible for reporting' = Financial institution with reporting obligation and involved in the transaction as Counterparty 1
	             'Counterparty 1 (reporting counterparty)' = Same as above
	             'Submitter identifier' = Same as above
	             [Case 2] Where a trading party is a fund:
	             'Entity responsible for reporting' = Trust Bank LEI
	             'Counterparty 1 (reporting counterparty)' = Fund LEI
	             'Submitter identifier' = Trust bank LEI 
	             [Case 3] Reporting party is an agent, etc. (where a regional bank requests the Trust Bank, etc. to report):
	             'Entity responsible for reporting' = Financial institution with reporting obligation and involved in the transaction as Counterparty 1
	             'Counterparty 1 (reporting counterparty)' = Same as above
	             'Submitter identifier' = LEI of the agent acting as the service provider for reporting
	 *
	 */
	@Override
	String getEntityResponsibleForReporting();
	/**
	 *
	 * Body CPMI_IOSCO
	 * Corpus TechnicalGuidance CDE Harmonisation of Critical Data Elements (other than UTI and UPI) version 3 "Version 3 of the CDE Technical Guidance. This new version of the CDE Technical Guidance (version 3) includes certain revisions and new data elements to the September 2021 CDE Technical Guidance. These proposed revisions and additions are deemed necessary to further improve the standardisation and understanding of the data."  * Corpus Dissemination Valuation   
	 * section "2" * field "8"
	 *
	 * Provision Indicator of whether LEI was used to identify the Counterparty 2
	             - Allowable Values:
	                 - true, for legal entities
	                 - false, for natural persons who are acting as private individuals (not eligible for an LEI per the ROC Statement - Individuals Acting in a Business Capacity.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220703"
	 *
	 * Provision Peer review recommended to detect private individual party representations (instead of LEI scheme). For natural persons who are acting as private individuals (not business entities): LEI of the reporting counterparty followed by a unique identifier assigned and maintained consistently by the reporting counterparty for that natural person(s) for regulatory reporting purpose.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20240606"
	 *
	 * Provision Peer Review agreed that a natural person involved as principal to a transaction can only appear as either the 'Buyer' or the 'Seller'. These roles are therefore used to identify natural persons acting in a private capacity. In such cases, Counterparty2IdentifierType must be set to False to indicate that the counterparty is not a legal entity with an LEI. For all other cases (legal entities with LEI), the value is set to True.
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "9" * field "Counterparty 2 Identifier Type"
	 *
	 * Provision Indicator of whether LEI was used to identify the Counterparty 2.
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "9" * field "Counterparty 2 Identifier Type - Remark"
	 *
	 * Provision Under the CDE Guidance, legal entity is to be reported as True and individual as False. However, for cases where a tentative LEI is used to report for a legal entity, it should be reported as False.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220307"
	 *
	 * Provision Peer review recommnended to detect private individual party representations (instead of LEI scheme).
	 *
	 */
	@Override
	Boolean getCounterparty2IdentifierType();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "12" * field "Submitter Identifier"
	 *
	 * Provision Identifier of the entity submitting the data to the swap data repository (SDR). The Submitter identifier will be the same as the reporting counterparty, unless they use a third-party service provider to submit the data to SDR in which case, report the identifier of the third-party service provider.
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "12" * field "Submitter Identifier - Remark"
	 *
	 * Provision See 'Entity responsible for reporting'. Applicable for central counterparty and financial institutions. If reporting is conducted using a platform, then the platform is deemed to be the submitter.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220309"
	 *
	 * Provision Check whether trade is executed in a SEF first. If it is, SEF is obliged to be the Submitter. Then check for Reporting Party and Data Submitter.
	 *
	 */
	@Override
	String getReportSubmittingEntityID();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "[Internal]" * field "Technical Record Id (ISO)"
	 *
	 * Provision Unique technical identification of the original data for which the status is provided for the BDR. Not sent to JFSA.
	 *
	 */
	@Override
	String getTechnicalRecordId();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "13" * field "New SDR Identifier"
	 *
	 * Provision Identifier of the new swap data repository where the swap transaction is transferred to.
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "13" * field "New SDR Identifier - Remark"
	 *
	 * Provision If unapplicable, leave it blank.
	 *
	 */
	String getNewSDRIdentifier();

	/*********************** Build Methods  ***********************/
	JFSAValuationReport build();
	
	JFSAValuationReport.JFSAValuationReportBuilder toBuilder();
	
	static JFSAValuationReport.JFSAValuationReportBuilder builder() {
		return new JFSAValuationReport.JFSAValuationReportBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends JFSAValuationReport> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends JFSAValuationReport> getType() {
		return JFSAValuationReport.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processor.processBasic(path.newSubPath("reportingTimestamp"), ZonedDateTime.class, getReportingTimestamp(), this);
		processor.processBasic(path.newSubPath("counterparty1"), String.class, getCounterparty1(), this);
		processor.processBasic(path.newSubPath("counterparty2IdentifierType"), Boolean.class, getCounterparty2IdentifierType(), this);
		processor.processBasic(path.newSubPath("counterparty2"), String.class, getCounterparty2(), this);
		processor.processBasic(path.newSubPath("valuationAmount"), BigDecimal.class, getValuationAmount(), this);
		processor.processBasic(path.newSubPath("valuationCurrency"), ISOCurrencyCodeEnum.class, getValuationCurrency(), this);
		processor.processBasic(path.newSubPath("valuationTimestamp"), ZonedDateTime.class, getValuationTimestamp(), this);
		processor.processBasic(path.newSubPath("valuationMethod"), ValuationType1Code.class, getValuationMethod(), this);
		processor.processBasic(path.newSubPath("delta"), BigDecimal.class, getDelta(), this);
		processor.processBasic(path.newSubPath("actionType"), ActionTypeEnum.class, getActionType(), this);
		processor.processBasic(path.newSubPath("level"), ReportLevelEnum.class, getLevel(), this);
		processor.processBasic(path.newSubPath("technicalRecordId"), String.class, getTechnicalRecordId(), this);
		processor.processBasic(path.newSubPath("reportSubmittingEntityID"), String.class, getReportSubmittingEntityID(), this);
		processor.processBasic(path.newSubPath("entityResponsibleForReporting"), String.class, getEntityResponsibleForReporting(), this);
		processor.processBasic(path.newSubPath("counterparty2IdentifierSource"), String.class, getCounterparty2IdentifierSource(), this);
		processor.processBasic(path.newSubPath("nextFloatingReferenceResetDateLeg1"), Date.class, getNextFloatingReferenceResetDateLeg1(), this);
		processor.processBasic(path.newSubPath("nextFloatingReferenceResetDateLeg2"), Date.class, getNextFloatingReferenceResetDateLeg2(), this);
		processor.processBasic(path.newSubPath("uniqueTransactionIdentifier"), String.class, getUniqueTransactionIdentifier(), this);
		processor.processBasic(path.newSubPath("uniqueTransactionIdentifierProprietary"), String.class, getUniqueTransactionIdentifierProprietary(), this);
		processor.processBasic(path.newSubPath("eventDate"), Date.class, getEventDate(), this);
		processor.processBasic(path.newSubPath("variationMarginCollateralPortfolioCode"), String.class, getVariationMarginCollateralPortfolioCode(), this);
		processor.processBasic(path.newSubPath("initialMarginCollateralPortfolioCode"), String.class, getInitialMarginCollateralPortfolioCode(), this);
		processor.processBasic(path.newSubPath("lastFloatingReferenceValueLeg1"), BigDecimal.class, getLastFloatingReferenceValueLeg1(), this);
		processor.processBasic(path.newSubPath("lastFloatingReferenceValueLeg2"), BigDecimal.class, getLastFloatingReferenceValueLeg2(), this);
		processor.processBasic(path.newSubPath("lastFloatingReferenceResetDateLeg1"), Date.class, getLastFloatingReferenceResetDateLeg1(), this);
		processor.processBasic(path.newSubPath("lastFloatingReferenceResetDateLeg2"), Date.class, getLastFloatingReferenceResetDateLeg2(), this);
		processor.processBasic(path.newSubPath("newSDRIdentifier"), String.class, getNewSDRIdentifier(), this);
	}
	

	/*********************** Builder Interface  ***********************/
	interface JFSAValuationReportBuilder extends JFSAValuationReport, CommonValuationReport.CommonValuationReportBuilder {
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setReportingTimestamp(ZonedDateTime reportingTimestamp);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setCounterparty1(String counterparty1);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setCounterparty2IdentifierType(Boolean counterparty2IdentifierType);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setCounterparty2(String counterparty2);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setValuationAmount(BigDecimal valuationAmount);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setValuationCurrency(ISOCurrencyCodeEnum valuationCurrency);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setValuationTimestamp(ZonedDateTime valuationTimestamp);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setValuationMethod(ValuationType1Code valuationMethod);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setDelta(BigDecimal delta);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setActionType(ActionTypeEnum actionType);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setLevel(ReportLevelEnum level);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setTechnicalRecordId(String technicalRecordId);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setReportSubmittingEntityID(String reportSubmittingEntityID);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setEntityResponsibleForReporting(String entityResponsibleForReporting);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setCounterparty2IdentifierSource(String counterparty2IdentifierSource);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setNextFloatingReferenceResetDateLeg1(Date nextFloatingReferenceResetDateLeg1);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setNextFloatingReferenceResetDateLeg2(Date nextFloatingReferenceResetDateLeg2);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setUniqueTransactionIdentifier(String uniqueTransactionIdentifier);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setUniqueTransactionIdentifierProprietary(String uniqueTransactionIdentifierProprietary);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setEventDate(Date eventDate);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setVariationMarginCollateralPortfolioCode(String variationMarginCollateralPortfolioCode);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setInitialMarginCollateralPortfolioCode(String initialMarginCollateralPortfolioCode);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setLastFloatingReferenceValueLeg1(BigDecimal lastFloatingReferenceValueLeg1);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setLastFloatingReferenceValueLeg2(BigDecimal lastFloatingReferenceValueLeg2);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setLastFloatingReferenceResetDateLeg1(Date lastFloatingReferenceResetDateLeg1);
		@Override
		JFSAValuationReport.JFSAValuationReportBuilder setLastFloatingReferenceResetDateLeg2(Date lastFloatingReferenceResetDateLeg2);
		JFSAValuationReport.JFSAValuationReportBuilder setEntityResponsibleForReportingOverriddenAsString(String entityResponsibleForReporting);
		JFSAValuationReport.JFSAValuationReportBuilder setCounterparty2IdentifierTypeOverriddenAsBoolean(Boolean counterparty2IdentifierType);
		JFSAValuationReport.JFSAValuationReportBuilder setReportSubmittingEntityIDOverriddenAsString(String reportSubmittingEntityID);
		JFSAValuationReport.JFSAValuationReportBuilder setTechnicalRecordIdOverriddenAsString(String technicalRecordId);
		JFSAValuationReport.JFSAValuationReportBuilder setNewSDRIdentifier(String newSDRIdentifier);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processor.processBasic(path.newSubPath("reportingTimestamp"), ZonedDateTime.class, getReportingTimestamp(), this);
			processor.processBasic(path.newSubPath("counterparty1"), String.class, getCounterparty1(), this);
			processor.processBasic(path.newSubPath("counterparty2IdentifierType"), Boolean.class, getCounterparty2IdentifierType(), this);
			processor.processBasic(path.newSubPath("counterparty2"), String.class, getCounterparty2(), this);
			processor.processBasic(path.newSubPath("valuationAmount"), BigDecimal.class, getValuationAmount(), this);
			processor.processBasic(path.newSubPath("valuationCurrency"), ISOCurrencyCodeEnum.class, getValuationCurrency(), this);
			processor.processBasic(path.newSubPath("valuationTimestamp"), ZonedDateTime.class, getValuationTimestamp(), this);
			processor.processBasic(path.newSubPath("valuationMethod"), ValuationType1Code.class, getValuationMethod(), this);
			processor.processBasic(path.newSubPath("delta"), BigDecimal.class, getDelta(), this);
			processor.processBasic(path.newSubPath("actionType"), ActionTypeEnum.class, getActionType(), this);
			processor.processBasic(path.newSubPath("level"), ReportLevelEnum.class, getLevel(), this);
			processor.processBasic(path.newSubPath("technicalRecordId"), String.class, getTechnicalRecordId(), this);
			processor.processBasic(path.newSubPath("reportSubmittingEntityID"), String.class, getReportSubmittingEntityID(), this);
			processor.processBasic(path.newSubPath("entityResponsibleForReporting"), String.class, getEntityResponsibleForReporting(), this);
			processor.processBasic(path.newSubPath("counterparty2IdentifierSource"), String.class, getCounterparty2IdentifierSource(), this);
			processor.processBasic(path.newSubPath("nextFloatingReferenceResetDateLeg1"), Date.class, getNextFloatingReferenceResetDateLeg1(), this);
			processor.processBasic(path.newSubPath("nextFloatingReferenceResetDateLeg2"), Date.class, getNextFloatingReferenceResetDateLeg2(), this);
			processor.processBasic(path.newSubPath("uniqueTransactionIdentifier"), String.class, getUniqueTransactionIdentifier(), this);
			processor.processBasic(path.newSubPath("uniqueTransactionIdentifierProprietary"), String.class, getUniqueTransactionIdentifierProprietary(), this);
			processor.processBasic(path.newSubPath("eventDate"), Date.class, getEventDate(), this);
			processor.processBasic(path.newSubPath("variationMarginCollateralPortfolioCode"), String.class, getVariationMarginCollateralPortfolioCode(), this);
			processor.processBasic(path.newSubPath("initialMarginCollateralPortfolioCode"), String.class, getInitialMarginCollateralPortfolioCode(), this);
			processor.processBasic(path.newSubPath("lastFloatingReferenceValueLeg1"), BigDecimal.class, getLastFloatingReferenceValueLeg1(), this);
			processor.processBasic(path.newSubPath("lastFloatingReferenceValueLeg2"), BigDecimal.class, getLastFloatingReferenceValueLeg2(), this);
			processor.processBasic(path.newSubPath("lastFloatingReferenceResetDateLeg1"), Date.class, getLastFloatingReferenceResetDateLeg1(), this);
			processor.processBasic(path.newSubPath("lastFloatingReferenceResetDateLeg2"), Date.class, getLastFloatingReferenceResetDateLeg2(), this);
			processor.processBasic(path.newSubPath("newSDRIdentifier"), String.class, getNewSDRIdentifier(), this);
		}
		

		JFSAValuationReport.JFSAValuationReportBuilder prune();
	}

	/*********************** Immutable Implementation of JFSAValuationReport  ***********************/
	class JFSAValuationReportImpl extends CommonValuationReport.CommonValuationReportImpl implements JFSAValuationReport {
		private final String entityResponsibleForReporting;
		private final Boolean counterparty2IdentifierType;
		private final String reportSubmittingEntityID;
		private final String technicalRecordId;
		private final String newSDRIdentifier;
		
		protected JFSAValuationReportImpl(JFSAValuationReport.JFSAValuationReportBuilder builder) {
			super(builder);
			this.entityResponsibleForReporting = builder.getEntityResponsibleForReporting();
			this.counterparty2IdentifierType = builder.getCounterparty2IdentifierType();
			this.reportSubmittingEntityID = builder.getReportSubmittingEntityID();
			this.technicalRecordId = builder.getTechnicalRecordId();
			this.newSDRIdentifier = builder.getNewSDRIdentifier();
		}
		
		@Override
		@RosettaAttribute(value="entityResponsibleForReporting", isRequired=true)
		@RuneAttribute(value="entityResponsibleForReporting", isRequired=true)
		public String getEntityResponsibleForReporting() {
			return entityResponsibleForReporting;
		}
		
		@Override
		@RosettaAttribute(value="counterparty2IdentifierType", isRequired=true)
		@RuneAttribute(value="counterparty2IdentifierType", isRequired=true)
		public Boolean getCounterparty2IdentifierType() {
			return counterparty2IdentifierType;
		}
		
		@Override
		@RosettaAttribute(value="reportSubmittingEntityID", isRequired=true)
		@RuneAttribute(value="reportSubmittingEntityID", isRequired=true)
		public String getReportSubmittingEntityID() {
			return reportSubmittingEntityID;
		}
		
		@Override
		@RosettaAttribute(value="technicalRecordId", isRequired=true)
		@RuneAttribute(value="technicalRecordId", isRequired=true)
		public String getTechnicalRecordId() {
			return technicalRecordId;
		}
		
		@Override
		@RosettaAttribute("newSDRIdentifier")
		@RuneAttribute("newSDRIdentifier")
		public String getNewSDRIdentifier() {
			return newSDRIdentifier;
		}
		
		@Override
		public JFSAValuationReport build() {
			return this;
		}
		
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder toBuilder() {
			JFSAValuationReport.JFSAValuationReportBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(JFSAValuationReport.JFSAValuationReportBuilder builder) {
			super.setBuilderFields(builder);
			ofNullable(getEntityResponsibleForReporting()).ifPresent(builder::setEntityResponsibleForReportingOverriddenAsString);
			ofNullable(getCounterparty2IdentifierType()).ifPresent(builder::setCounterparty2IdentifierTypeOverriddenAsBoolean);
			ofNullable(getReportSubmittingEntityID()).ifPresent(builder::setReportSubmittingEntityIDOverriddenAsString);
			ofNullable(getTechnicalRecordId()).ifPresent(builder::setTechnicalRecordIdOverriddenAsString);
			ofNullable(getNewSDRIdentifier()).ifPresent(builder::setNewSDRIdentifier);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			JFSAValuationReport _that = getType().cast(o);
		
			if (!Objects.equals(entityResponsibleForReporting, _that.getEntityResponsibleForReporting())) return false;
			if (!Objects.equals(counterparty2IdentifierType, _that.getCounterparty2IdentifierType())) return false;
			if (!Objects.equals(reportSubmittingEntityID, _that.getReportSubmittingEntityID())) return false;
			if (!Objects.equals(technicalRecordId, _that.getTechnicalRecordId())) return false;
			if (!Objects.equals(newSDRIdentifier, _that.getNewSDRIdentifier())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (entityResponsibleForReporting != null ? entityResponsibleForReporting.hashCode() : 0);
			_result = 31 * _result + (counterparty2IdentifierType != null ? counterparty2IdentifierType.hashCode() : 0);
			_result = 31 * _result + (reportSubmittingEntityID != null ? reportSubmittingEntityID.hashCode() : 0);
			_result = 31 * _result + (technicalRecordId != null ? technicalRecordId.hashCode() : 0);
			_result = 31 * _result + (newSDRIdentifier != null ? newSDRIdentifier.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "JFSAValuationReport {" +
				"entityResponsibleForReporting=" + this.entityResponsibleForReporting + ", " +
				"counterparty2IdentifierType=" + this.counterparty2IdentifierType + ", " +
				"reportSubmittingEntityID=" + this.reportSubmittingEntityID + ", " +
				"technicalRecordId=" + this.technicalRecordId + ", " +
				"newSDRIdentifier=" + this.newSDRIdentifier +
			'}' + " " + super.toString();
		}
	}

	/*********************** Builder Implementation of JFSAValuationReport  ***********************/
	class JFSAValuationReportBuilderImpl extends CommonValuationReport.CommonValuationReportBuilderImpl implements JFSAValuationReport.JFSAValuationReportBuilder {
	
		protected String entityResponsibleForReporting;
		protected Boolean counterparty2IdentifierType;
		protected String reportSubmittingEntityID;
		protected String technicalRecordId;
		protected String newSDRIdentifier;
		
		@Override
		@RosettaAttribute(value="entityResponsibleForReporting", isRequired=true)
		@RuneAttribute(value="entityResponsibleForReporting", isRequired=true)
		public String getEntityResponsibleForReporting() {
			return entityResponsibleForReporting;
		}
		
		@Override
		@RosettaAttribute(value="counterparty2IdentifierType", isRequired=true)
		@RuneAttribute(value="counterparty2IdentifierType", isRequired=true)
		public Boolean getCounterparty2IdentifierType() {
			return counterparty2IdentifierType;
		}
		
		@Override
		@RosettaAttribute(value="reportSubmittingEntityID", isRequired=true)
		@RuneAttribute(value="reportSubmittingEntityID", isRequired=true)
		public String getReportSubmittingEntityID() {
			return reportSubmittingEntityID;
		}
		
		@Override
		@RosettaAttribute(value="technicalRecordId", isRequired=true)
		@RuneAttribute(value="technicalRecordId", isRequired=true)
		public String getTechnicalRecordId() {
			return technicalRecordId;
		}
		
		@Override
		@RosettaAttribute("newSDRIdentifier")
		@RuneAttribute("newSDRIdentifier")
		public String getNewSDRIdentifier() {
			return newSDRIdentifier;
		}
		
		@RosettaAttribute(value="reportingTimestamp", isRequired=true)
		@RuneAttribute(value="reportingTimestamp", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setReportingTimestamp(ZonedDateTime _reportingTimestamp) {
			this.reportingTimestamp = _reportingTimestamp == null ? null : _reportingTimestamp;
			return this;
		}
		
		@RosettaAttribute(value="counterparty1", isRequired=true)
		@RuneAttribute(value="counterparty1", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setCounterparty1(String _counterparty1) {
			this.counterparty1 = _counterparty1 == null ? null : _counterparty1;
			return this;
		}
		
		@RosettaAttribute(value="counterparty2IdentifierType", isRequired=true)
		@RuneAttribute(value="counterparty2IdentifierType", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setCounterparty2IdentifierTypeOverriddenAsBoolean(Boolean _counterparty2IdentifierType) {
			this.counterparty2IdentifierType = _counterparty2IdentifierType == null ? null : _counterparty2IdentifierType;
			return this;
		}
		
		@RosettaIgnore
		@RuneIgnore
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setCounterparty2IdentifierType(Boolean _counterparty2IdentifierType) {
			return setCounterparty2IdentifierTypeOverriddenAsBoolean(_counterparty2IdentifierType);
		}
		
		@RosettaAttribute(value="counterparty2", isRequired=true)
		@RuneAttribute(value="counterparty2", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setCounterparty2(String _counterparty2) {
			this.counterparty2 = _counterparty2 == null ? null : _counterparty2;
			return this;
		}
		
		@RosettaAttribute(value="valuationAmount", isRequired=true)
		@RuneAttribute(value="valuationAmount", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setValuationAmount(BigDecimal _valuationAmount) {
			this.valuationAmount = _valuationAmount == null ? null : _valuationAmount;
			return this;
		}
		
		@RosettaAttribute(value="valuationCurrency", isRequired=true)
		@RuneAttribute(value="valuationCurrency", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setValuationCurrency(ISOCurrencyCodeEnum _valuationCurrency) {
			this.valuationCurrency = _valuationCurrency == null ? null : _valuationCurrency;
			return this;
		}
		
		@RosettaAttribute(value="valuationTimestamp", isRequired=true)
		@RuneAttribute(value="valuationTimestamp", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setValuationTimestamp(ZonedDateTime _valuationTimestamp) {
			this.valuationTimestamp = _valuationTimestamp == null ? null : _valuationTimestamp;
			return this;
		}
		
		@RosettaAttribute(value="valuationMethod", isRequired=true)
		@RuneAttribute(value="valuationMethod", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setValuationMethod(ValuationType1Code _valuationMethod) {
			this.valuationMethod = _valuationMethod == null ? null : _valuationMethod;
			return this;
		}
		
		@RosettaAttribute("delta")
		@RuneAttribute("delta")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setDelta(BigDecimal _delta) {
			this.delta = _delta == null ? null : _delta;
			return this;
		}
		
		@RosettaAttribute(value="actionType", isRequired=true)
		@RuneAttribute(value="actionType", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setActionType(ActionTypeEnum _actionType) {
			this.actionType = _actionType == null ? null : _actionType;
			return this;
		}
		
		@RosettaAttribute("level")
		@RuneAttribute("level")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setLevel(ReportLevelEnum _level) {
			this.level = _level == null ? null : _level;
			return this;
		}
		
		@RosettaAttribute(value="technicalRecordId", isRequired=true)
		@RuneAttribute(value="technicalRecordId", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setTechnicalRecordIdOverriddenAsString(String _technicalRecordId) {
			this.technicalRecordId = _technicalRecordId == null ? null : _technicalRecordId;
			return this;
		}
		
		@RosettaIgnore
		@RuneIgnore
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setTechnicalRecordId(String _technicalRecordId) {
			return setTechnicalRecordIdOverriddenAsString(_technicalRecordId);
		}
		
		@RosettaAttribute(value="reportSubmittingEntityID", isRequired=true)
		@RuneAttribute(value="reportSubmittingEntityID", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setReportSubmittingEntityIDOverriddenAsString(String _reportSubmittingEntityID) {
			this.reportSubmittingEntityID = _reportSubmittingEntityID == null ? null : _reportSubmittingEntityID;
			return this;
		}
		
		@RosettaIgnore
		@RuneIgnore
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setReportSubmittingEntityID(String _reportSubmittingEntityID) {
			return setReportSubmittingEntityIDOverriddenAsString(_reportSubmittingEntityID);
		}
		
		@RosettaAttribute(value="entityResponsibleForReporting", isRequired=true)
		@RuneAttribute(value="entityResponsibleForReporting", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setEntityResponsibleForReportingOverriddenAsString(String _entityResponsibleForReporting) {
			this.entityResponsibleForReporting = _entityResponsibleForReporting == null ? null : _entityResponsibleForReporting;
			return this;
		}
		
		@RosettaIgnore
		@RuneIgnore
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setEntityResponsibleForReporting(String _entityResponsibleForReporting) {
			return setEntityResponsibleForReportingOverriddenAsString(_entityResponsibleForReporting);
		}
		
		@RosettaAttribute("counterparty2IdentifierSource")
		@RuneAttribute("counterparty2IdentifierSource")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setCounterparty2IdentifierSource(String _counterparty2IdentifierSource) {
			this.counterparty2IdentifierSource = _counterparty2IdentifierSource == null ? null : _counterparty2IdentifierSource;
			return this;
		}
		
		@RosettaAttribute("nextFloatingReferenceResetDateLeg1")
		@RuneAttribute("nextFloatingReferenceResetDateLeg1")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setNextFloatingReferenceResetDateLeg1(Date _nextFloatingReferenceResetDateLeg1) {
			this.nextFloatingReferenceResetDateLeg1 = _nextFloatingReferenceResetDateLeg1 == null ? null : _nextFloatingReferenceResetDateLeg1;
			return this;
		}
		
		@RosettaAttribute("nextFloatingReferenceResetDateLeg2")
		@RuneAttribute("nextFloatingReferenceResetDateLeg2")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setNextFloatingReferenceResetDateLeg2(Date _nextFloatingReferenceResetDateLeg2) {
			this.nextFloatingReferenceResetDateLeg2 = _nextFloatingReferenceResetDateLeg2 == null ? null : _nextFloatingReferenceResetDateLeg2;
			return this;
		}
		
		@RosettaAttribute(value="uniqueTransactionIdentifier", isRequired=true)
		@RuneAttribute(value="uniqueTransactionIdentifier", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setUniqueTransactionIdentifier(String _uniqueTransactionIdentifier) {
			this.uniqueTransactionIdentifier = _uniqueTransactionIdentifier == null ? null : _uniqueTransactionIdentifier;
			return this;
		}
		
		@RosettaAttribute("uniqueTransactionIdentifierProprietary")
		@RuneAttribute("uniqueTransactionIdentifierProprietary")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setUniqueTransactionIdentifierProprietary(String _uniqueTransactionIdentifierProprietary) {
			this.uniqueTransactionIdentifierProprietary = _uniqueTransactionIdentifierProprietary == null ? null : _uniqueTransactionIdentifierProprietary;
			return this;
		}
		
		@RosettaAttribute("eventDate")
		@RuneAttribute("eventDate")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setEventDate(Date _eventDate) {
			this.eventDate = _eventDate == null ? null : _eventDate;
			return this;
		}
		
		@RosettaAttribute("variationMarginCollateralPortfolioCode")
		@RuneAttribute("variationMarginCollateralPortfolioCode")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setVariationMarginCollateralPortfolioCode(String _variationMarginCollateralPortfolioCode) {
			this.variationMarginCollateralPortfolioCode = _variationMarginCollateralPortfolioCode == null ? null : _variationMarginCollateralPortfolioCode;
			return this;
		}
		
		@RosettaAttribute("initialMarginCollateralPortfolioCode")
		@RuneAttribute("initialMarginCollateralPortfolioCode")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setInitialMarginCollateralPortfolioCode(String _initialMarginCollateralPortfolioCode) {
			this.initialMarginCollateralPortfolioCode = _initialMarginCollateralPortfolioCode == null ? null : _initialMarginCollateralPortfolioCode;
			return this;
		}
		
		@RosettaAttribute("lastFloatingReferenceValueLeg1")
		@RuneAttribute("lastFloatingReferenceValueLeg1")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setLastFloatingReferenceValueLeg1(BigDecimal _lastFloatingReferenceValueLeg1) {
			this.lastFloatingReferenceValueLeg1 = _lastFloatingReferenceValueLeg1 == null ? null : _lastFloatingReferenceValueLeg1;
			return this;
		}
		
		@RosettaAttribute("lastFloatingReferenceValueLeg2")
		@RuneAttribute("lastFloatingReferenceValueLeg2")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setLastFloatingReferenceValueLeg2(BigDecimal _lastFloatingReferenceValueLeg2) {
			this.lastFloatingReferenceValueLeg2 = _lastFloatingReferenceValueLeg2 == null ? null : _lastFloatingReferenceValueLeg2;
			return this;
		}
		
		@RosettaAttribute("lastFloatingReferenceResetDateLeg1")
		@RuneAttribute("lastFloatingReferenceResetDateLeg1")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setLastFloatingReferenceResetDateLeg1(Date _lastFloatingReferenceResetDateLeg1) {
			this.lastFloatingReferenceResetDateLeg1 = _lastFloatingReferenceResetDateLeg1 == null ? null : _lastFloatingReferenceResetDateLeg1;
			return this;
		}
		
		@RosettaAttribute("lastFloatingReferenceResetDateLeg2")
		@RuneAttribute("lastFloatingReferenceResetDateLeg2")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setLastFloatingReferenceResetDateLeg2(Date _lastFloatingReferenceResetDateLeg2) {
			this.lastFloatingReferenceResetDateLeg2 = _lastFloatingReferenceResetDateLeg2 == null ? null : _lastFloatingReferenceResetDateLeg2;
			return this;
		}
		
		@RosettaAttribute("newSDRIdentifier")
		@RuneAttribute("newSDRIdentifier")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setNewSDRIdentifier(String _newSDRIdentifier) {
			this.newSDRIdentifier = _newSDRIdentifier == null ? null : _newSDRIdentifier;
			return this;
		}
		
		@Override
		public JFSAValuationReport build() {
			return new JFSAValuationReport.JFSAValuationReportImpl(this);
		}
		
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder prune() {
			super.prune();
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (super.hasData()) return true;
			if (getEntityResponsibleForReporting()!=null) return true;
			if (getCounterparty2IdentifierType()!=null) return true;
			if (getReportSubmittingEntityID()!=null) return true;
			if (getTechnicalRecordId()!=null) return true;
			if (getNewSDRIdentifier()!=null) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			super.merge(other, merger);
			JFSAValuationReport.JFSAValuationReportBuilder o = (JFSAValuationReport.JFSAValuationReportBuilder) other;
			
			
			merger.mergeBasic(getEntityResponsibleForReporting(), o.getEntityResponsibleForReporting(), this::setEntityResponsibleForReportingOverriddenAsString);
			merger.mergeBasic(getCounterparty2IdentifierType(), o.getCounterparty2IdentifierType(), this::setCounterparty2IdentifierTypeOverriddenAsBoolean);
			merger.mergeBasic(getReportSubmittingEntityID(), o.getReportSubmittingEntityID(), this::setReportSubmittingEntityIDOverriddenAsString);
			merger.mergeBasic(getTechnicalRecordId(), o.getTechnicalRecordId(), this::setTechnicalRecordIdOverriddenAsString);
			merger.mergeBasic(getNewSDRIdentifier(), o.getNewSDRIdentifier(), this::setNewSDRIdentifier);
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			JFSAValuationReport _that = getType().cast(o);
		
			if (!Objects.equals(entityResponsibleForReporting, _that.getEntityResponsibleForReporting())) return false;
			if (!Objects.equals(counterparty2IdentifierType, _that.getCounterparty2IdentifierType())) return false;
			if (!Objects.equals(reportSubmittingEntityID, _that.getReportSubmittingEntityID())) return false;
			if (!Objects.equals(technicalRecordId, _that.getTechnicalRecordId())) return false;
			if (!Objects.equals(newSDRIdentifier, _that.getNewSDRIdentifier())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (entityResponsibleForReporting != null ? entityResponsibleForReporting.hashCode() : 0);
			_result = 31 * _result + (counterparty2IdentifierType != null ? counterparty2IdentifierType.hashCode() : 0);
			_result = 31 * _result + (reportSubmittingEntityID != null ? reportSubmittingEntityID.hashCode() : 0);
			_result = 31 * _result + (technicalRecordId != null ? technicalRecordId.hashCode() : 0);
			_result = 31 * _result + (newSDRIdentifier != null ? newSDRIdentifier.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "JFSAValuationReportBuilder {" +
				"entityResponsibleForReporting=" + this.entityResponsibleForReporting + ", " +
				"counterparty2IdentifierType=" + this.counterparty2IdentifierType + ", " +
				"reportSubmittingEntityID=" + this.reportSubmittingEntityID + ", " +
				"technicalRecordId=" + this.technicalRecordId + ", " +
				"newSDRIdentifier=" + this.newSDRIdentifier +
			'}' + " " + super.toString();
		}
	}
}
