package org.isda.drr.example.reporting.collateralmargin;

import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.asic.rewrite.margin.functions.Project_ASICMarginReportToIso20022;
import drr.regulation.asic.rewrite.margin.ASICMarginReport;
import drr.regulation.asic.rewrite.margin.reports.ASICMarginReportFunction;
import iso20022.Auth108AsicModelConfig;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate ASIC collateral reporting with ISO-20022 projection.
 * This class validates the transformation of a `CollateralReportInstruction` to an `ASICMarginReport`,
 * followed by validation and ISO-20022 projection.
 */
final class ASICCollateralWithISOProjectionTest extends AbstractReportingTest {

    // Function to generate an ASIC Margin Report from a CollateralReportInstruction
    @Inject
    ASICMarginReportFunction reportFunc;

    // Function to project an ASIC Margin Report to ISO-20022 format
    @Inject
    Project_ASICMarginReportToIso20022 asicMarginReportToIso20022;

    /**
     * Demonstrates ASIC collateral reporting with ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `CollateralReportInstruction` from JSON.
     * - Generates an `ASICMarginReport` and validates it.
     * - Projects the report to ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void ASICCollateralWithISOProjectionExampleTest() throws IOException {

        // Generate the ASIC margin report
        ASICMarginReport report = runReport(collateralReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(ASICMarginReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to ISO-20022 format
        projectReport(report, asicMarginReportToIso20022::evaluate, Auth108AsicModelConfig.XML_CONFIG_PATH);
    }
}
