package org.isda.drr.example.reporting.collateralmargin;

import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.mas.rewrite.margin.functions.Project_MASMarginReportToIso20022;
import drr.regulation.mas.rewrite.margin.MASMarginReport;
import drr.regulation.mas.rewrite.margin.reports.MASMarginReportFunction;
import iso20022.Auth108MasModelConfig;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate MAS collateral reporting with ISO-20022 projection.
 * This class validates the transformation of a `CollateralReportInstruction` to a `MASMarginReport`,
 * followed by validation and ISO-20022 projection.
 */
final class MASCollateralWithISOProjectionTest extends AbstractReportingTest {

    // Function to generate a MAS Margin Report from a CollateralReportInstruction
    @Inject
    MASMarginReportFunction reportFunc;

    // Function to project a MAS Margin Report to ISO-20022 format
    @Inject
    Project_MASMarginReportToIso20022 masMarginReportToIso20022;

    /**
     * Demonstrates MAS collateral reporting with ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `CollateralReportInstruction` from JSON.
     * - Generates a `MASMarginReport` and validates it.
     * - Projects the report to ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void MASCollateralWithISOProjectionExampleTest() throws IOException {

        // Generate the MAS margin report
        MASMarginReport report = runReport(collateralReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(MASMarginReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to ISO-20022 format
        projectReport(report, masMarginReportToIso20022::evaluate, Auth108MasModelConfig.XML_CONFIG_PATH);
    }
}
