package org.isda.drr.example.reporting.transaction;

import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.fca.ukemir.refit.trade.functions.Project_FcaUkEmirTradeReportToIso20022;
import drr.regulation.fca.ukemir.refit.trade.FCAUKEMIRTransactionReport;
import drr.regulation.fca.ukemir.refit.trade.reports.FCAUKEMIRTradeReportFunction;
import iso20022.Auth030FcaModelConfig;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate UK EMIR trade reporting with ISO-20022 projection.
 * This class validates the transformation of a `TransactionReportInstruction` to a `FCAUKEMIRTransactionReport`,
 * followed by validation and ISO-20022 projection.
 */
final class UKEMIRTradeWithISOProjectionTest extends AbstractReportingTest {

    // Function to generate a UK EMIR Trade Report from a TransactionReportInstruction
    @Inject
    FCAUKEMIRTradeReportFunction reportFunc;

    // Function to project a UK EMIR Trade Report to ISO-20022 format
    @Inject
    Project_FcaUkEmirTradeReportToIso20022 ukEmirTradeReportToIso20022;

    /**
     * Demonstrates UK EMIR trade reporting with ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `TransactionReportInstruction` from JSON.
     * - Generates a `FCAUKEMIRTransactionReport` and validates it.
     * - Projects the report to ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void UKEMIRTradeWithISOProjectionExampleTest() throws IOException {

        // Generate the UK EMIR transaction report
        FCAUKEMIRTransactionReport report = runReport(transactionReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(FCAUKEMIRTransactionReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to ISO-20022 format
        projectReport(report, ukEmirTradeReportToIso20022::evaluate, Auth030FcaModelConfig.XML_CONFIG_PATH);
    }
}
