package org.isda.drr.example.reporting.valuation;

import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.asic.rewrite.valuation.functions.Project_ASICValuationReportToIso20022;
import drr.regulation.asic.rewrite.valuation.ASICValuationReport;
import drr.regulation.asic.rewrite.valuation.reports.ASICValuationReportFunction;
import iso20022.Auth030AsicModelConfig;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate ASIC valuation reporting with ISO-20022 projection.
 * The test validates the transformation of a `ValuationReportInstruction` to an `ASICValuationReport`,
 * followed by validation and ISO-20022 projection.
 */
final class ASICValuationWithISOProjectionTest extends AbstractReportingTest {

    // Function to generate an ASICValuationReport from a ValuationReportInstruction.
    @Inject
    ASICValuationReportFunction reportFunc;

    @Inject
    Project_ASICValuationReportToIso20022 asicValuationReportToIso20022;

    /**
     * Demonstrates ASIC valuation reporting with ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `ValuationReportInstruction` from JSON.
     * - Generates an `ASICValuationReport` and validates it.
     * - Projects the report to ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or writing.
     */
    @Test
    void ASICValuationWithISOProjectionExampleTest() throws IOException {

        // Generate the ASIC valuation report
        ASICValuationReport report = runReport(valuationReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(ASICValuationReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to ISO-20022 format.
        projectReport(report, asicValuationReportToIso20022::evaluate, Auth030AsicModelConfig.XML_CONFIG_PATH);
    }
}
