package org.isda.drr.example.reporting.transaction;

import com.google.common.io.Resources;
import jakarta.inject.Inject;
import com.regnosys.drr.examples.util.ResourcesUtils;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.asic.rewrite.trade.functions.Project_ASICTradeReportToIso20022;
import drr.regulation.asic.rewrite.trade.ASICTransactionReport;
import drr.regulation.asic.rewrite.trade.reports.ASICTradeReportFunction;
import drr.regulation.common.TransactionReportInstruction;
import iso20022.Auth030AsicModelConfig;
import iso20022.auth030.asic.Document;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.URL;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate ASIC trade reporting with ISO-20022 projection.
 * This class validates the transformation of a `TransactionReportInstruction` to an `ASICTransactionReport`,
 * followed by validation and ISO-20022 projection.
 */
final class ASICTradeWithISOProjectionTest extends AbstractReportingTest {

    private static final Logger logger = LoggerFactory.getLogger(ASICTradeWithISOProjectionTest.class);

    // Function to generate an ASIC Trade Report from a TransactionReportInstruction
    @Inject
    ASICTradeReportFunction reportFunc;

    // Function to project an ASIC Trade Report to ISO-20022 format
    @Inject
    Project_ASICTradeReportToIso20022 asicTradeReportToIso20022;

    /**
     * Demonstrates ASIC trade reporting with ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `TransactionReportInstruction` from JSON.
     * - Generates an `ASICTransactionReport` and validates it.
     * - Projects the report to ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void AsicTradeWithISOProjectionExampleTest() throws IOException {
        // Load a TransactionReportInstruction from the input test data
        TransactionReportInstruction transactionReportInstruction = ResourcesUtils.getObjectAndResolveReferences(
                TransactionReportInstruction.class,
                "regulatory-reporting/input/rates/IR-IRS-Fixed-Float-ex01.json"
        );
        assertNotNull(transactionReportInstruction, "No reportable event was found");

        // Generate the ASIC transaction report
        ASICTransactionReport report = runReport(transactionReportInstruction);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(ASICTransactionReport.class, report.toBuilder());
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to ISO-20022 format
        ISOProjection(report);
    }

    /**
     * Generates an ASIC transaction report from a `ReportableEvent`.
     *
     * @param transactionReportInstruction The transaction report instruction input.
     * @return The generated ASICTransactionReport.
     * @throws IOException If there is an error during processing.
     */
    ASICTransactionReport runReport(TransactionReportInstruction transactionReportInstruction) throws IOException {
        // Generate the ASIC transaction report
        final ASICTransactionReport report = reportFunc.evaluate(transactionReportInstruction);

        // Print the generated report in JSON format for debugging
        logger.debug(objectWriter.writeValueAsString(report));

        return report;
    }

    /**
     * Projects an `ASICTransactionReport` to ISO-20022 format.
     *
     * @param report The ASIC transaction report to project.
     * @throws IOException If there is an error during the projection process.
     */
    void ISOProjection(ASICTransactionReport report) throws IOException {
        // Project the ASIC transaction report to an ISO-20022 document
        Document iso20022Document = asicTradeReportToIso20022.evaluate(report);

        // Load the ISO-20022 configuration path
        URL iso20022Auth030XmlConfig = Resources.getResource(Auth030AsicModelConfig.XML_CONFIG_PATH);

        // Print the ISO-20022 document using the provided configuration
        ReportingTestUtils.logXMLProjection(iso20022Document, iso20022Auth030XmlConfig);
    }
}
