package org.isda.drr.example;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.google.common.io.Resources;
import com.regnosys.drr.examples.util.ResourcesUtils;
import drr.regulation.common.CollateralReportInstruction;
import drr.regulation.common.TransactionReportInstruction;
import drr.regulation.common.ValuationReportInstruction;
import jakarta.inject.Inject;
import com.regnosys.rosetta.common.hashing.ReferenceResolverProcessStep;
import com.regnosys.rosetta.common.serialisation.RosettaObjectMapper;
import com.regnosys.rosetta.common.validation.RosettaTypeValidator;
import com.regnosys.rosetta.common.validation.ValidationReport;
import com.rosetta.model.lib.RosettaModelObject;
import drr.enrichment.common.margin.functions.Create_CollateralReportInstruction;
import drr.enrichment.common.trade.functions.Create_TransactionReportInstruction;
import drr.enrichment.common.valuation.functions.Create_ValuationReportInstruction;
import org.isda.cdm.processor.CdmReferenceConfig;
import org.isda.drr.example.util.ReportingTestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.URL;
import java.util.function.Function;

/**
 * Abstract base class for reporting-related tests.
 * This class extends {@link AbstractExampleTest} and provides shared dependencies for
 * creating transaction, collateral, and valuation report instructions.
 * Subclasses can use these injected functions for their test implementations.
 */
public abstract class AbstractReportingTest extends AbstractExampleTest {

    @Inject
    protected RosettaTypeValidator validator;

    protected static final ObjectMapper mapper = RosettaObjectMapper.getNewRosettaObjectMapper();
    protected static final ObjectWriter objectWriter = mapper.writerWithDefaultPrettyPrinter();

    protected final Logger logger = LoggerFactory.getLogger(getClass());

    // Function to create transaction report instructions
    @Inject
    protected Create_TransactionReportInstruction createTransactionReportInstructionFunc;

    // Function to create collateral report instructions
    @Inject
    protected Create_CollateralReportInstruction createCollateralReportInstructionFunc;

    // Function to create valuation report instructions
    @Inject
    protected Create_ValuationReportInstruction createValuationReportInstruction;

    // Load a ValuationReportInstruction from the input test data.
    protected ValuationReportInstruction valuationReportInstruction;

    // Load a TransactionReportInstruction from the input test data
    protected TransactionReportInstruction transactionReportInstruction;

    // Load a CollateralReportInstruction from the input test data
    protected CollateralReportInstruction collateralReportInstruction;

    {
        try {
            // Load a ValuationReportInstruction from the input test data
            valuationReportInstruction = ResourcesUtils.getObjectAndResolveReferences(ValuationReportInstruction.class, "regulatory-reporting/input/valuation/Valuation-ex01.json");

            // Load a TransactionReportInstruction from the input test data
            transactionReportInstruction = ResourcesUtils.getObjectAndResolveReferences(TransactionReportInstruction.class, "regulatory-reporting/input/rates/IR-IRS-Fixed-Float-ex01.json");

            // Load a CollateralReportInstruction from the input test data
            collateralReportInstruction = ResourcesUtils.getObjectAndResolveReferences(CollateralReportInstruction.class, "regulatory-reporting/input/collateral/Collateral-ex01.json");
        } catch (IOException e) {
            throw new RuntimeException("Resource not found.", e);
        }
    }


    /**
     * Validates a Rosetta model object by applying CDM validation rules.
     * This method runs validation processes defined in the CDM framework to check
     * the correctness and consistency of the given object against its schema.
     *
     * @param input The Rosetta model object to be validated.
     * @param <T>   The type of the Rosetta model object.
     * @return A ValidationReport containing validation results, including errors
     *         and warnings, if any.
     */
    protected <T extends RosettaModelObject> ValidationReport validate(T input) {
        // Run validation on the input object using its type.
        return validator.runProcessStep(input.getClass(), input);
    }

    /**
     * Resolves references within a Rosetta model object to ensure consistency and completeness.
     * The resolution process ensures that all references within the object are valid and
     * points to the correct entities or objects in the model. This step is necessary
     * before validation or qualification in cases where objects have unresolved references.
     *
     * @param object The Rosetta model object with references to be resolved.
     * @param <T>    The type of the Rosetta model object.
     * @return A new instance of the Rosetta model object with resolved references.
     */
    protected static <T extends RosettaModelObject> T resolveReferences(T object) {
        // Convert the object to its builder representation for modification.
        RosettaModelObject builder = object.toBuilder();
        // Resolve references using the CDM reference resolution configuration.
        new ReferenceResolverProcessStep(CdmReferenceConfig.get()).runProcessStep(builder.getType(), builder);
        // Build and return the updated object with resolved references.
        return (T) builder.build();
    }

    /**
     * Runs a reporting function using the provided instruction, logs the generated result in JSON format,
     * and returns the resulting report object. This helper centralizes the common execution-and-logging pattern
     * used across multiple reporting jurisdictions.
     */
    protected <I, R> R runReport(I instruction, Function<I, R> reportFunc) throws IOException {
        // Execute the reporting function to generate the report
        R report = reportFunc.apply(instruction);

        // Log the generated report as JSON
        logger.debug(objectWriter.writeValueAsString(report));

        // Return the report
        return report;
    }

    /**
     * Projects a report into a specific ISO-20022 (or related) document format using a provided projector
     * function and logs the resulting XML. This helper encapsulates the common logic required to transform
     * a report into its XML representation, and it's used across multiple reporting jurisdictions.
     */
    protected <R, D extends RosettaModelObject> void projectReport(R report, Function<R, D> projector, String xmlConfigPath) throws IOException {
        // Transform the report into a specific document structure
        D document = projector.apply(report);

        // Load the XML projection configuration from the provided path
        URL config = Resources.getResource(xmlConfigPath);

        // Log  the generated XML projection using the configured serializer
        ReportingTestUtils.logXMLProjection(document, config);
    }
}

