package org.isda.drr.example.reporting.transaction;

import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.asic.rewrite.trade.functions.Project_ASICTradeReportToIso20022;
import drr.regulation.asic.rewrite.trade.ASICTransactionReport;
import drr.regulation.asic.rewrite.trade.reports.ASICTradeReportFunction;
import iso20022.Auth030AsicModelConfig;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate ASIC trade reporting with ISO-20022 projection.
 * This class validates the transformation of a `TransactionReportInstruction` to an `ASICTransactionReport`,
 * followed by validation and ISO-20022 projection.
 */
final class ASICTradeWithISOProjectionTest extends AbstractReportingTest {

    // Function to generate an ASIC Trade Report from a TransactionReportInstruction
    @Inject
    ASICTradeReportFunction reportFunc;

    // Function to project an ASIC Trade Report to ISO-20022 format
    @Inject
    Project_ASICTradeReportToIso20022 asicTradeReportToIso20022;

    /**
     * Demonstrates ASIC trade reporting with ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `TransactionReportInstruction` from JSON.
     * - Generates an `ASICTransactionReport` and validates it.
     * - Projects the report to ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void ASICTradeWithISOProjectionExampleTest() throws IOException {

        // Generate the ASIC transaction report
        ASICTransactionReport report = runReport(transactionReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(ASICTransactionReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to ISO-20022 format
        projectReport(report, asicTradeReportToIso20022::evaluate, Auth030AsicModelConfig.XML_CONFIG_PATH);
    }
}
