package org.isda.drr.example.reporting.transaction;

import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.jfsa.rewrite.trade.functions.Project_JFSARewriteTradeReportToIso20022;
import drr.regulation.jfsa.rewrite.trade.JFSATransactionReport;
import drr.regulation.jfsa.rewrite.trade.reports.JFSATradeReportFunction;
import iso20022.Auth030JfsaModelConfig;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate JFSA trade reporting with ISO-20022 projection.
 * This class validates the transformation of a `TransactionReportInstruction` to a `JFSATransactionReport`,
 * followed by validation and ISO-20022 projection.
 */
final class JFSATradeWithISOProjectionTest extends AbstractReportingTest {

    // Function to generate a JFSA Trade Report from a TransactionReportInstruction
    @Inject
    JFSATradeReportFunction reportFunc;

    // Function to project a JFSA Trade Report to ISO-20022 format
    @Inject
    Project_JFSARewriteTradeReportToIso20022 jfsaTradeReportToIso20022;

    /**
     * Demonstrates JFSA trade reporting with ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `TransactionReportInstruction` from JSON.
     * - Generates a `JFSATransactionReport` and validates it.
     * - Projects the report to ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void JFSATradeWithISOProjectionExampleTest() throws IOException {

        // Generate the JFSA transaction report
        JFSATransactionReport report = runReport(transactionReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(JFSATransactionReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to ISO-20022 format
        projectReport(report, jfsaTradeReportToIso20022::evaluate, Auth030JfsaModelConfig.XML_CONFIG_PATH);
    }
}
