package org.isda.drr.example.reporting.valuation;

import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.hkma.rewrite.valuation.dtcc.functions.Project_HKMADtccValuationReportToIso20022;
import drr.projection.iso20022.hkma.rewrite.valuation.tr.functions.Project_HKMATrValuationReportToIso20022;
import drr.regulation.hkma.rewrite.valuation.HKMAValuationReport;
import drr.regulation.hkma.rewrite.valuation.reports.HKMAValuationReportFunction;
import iso20022.Auth030HkmaDtccModelConfig;
import iso20022.Auth030HkmaTrModelConfig;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate HKMA valuation reporting with ISO-20022 projection.
 * The test validates the transformation of a `ValuationReportInstruction` to an `HKMAValuationReport`,
 * followed by validation and ISO-20022 projection.
 */
final class HKMAValuationWithISOProjectionTest extends AbstractReportingTest {

    // Function to generate an HKMAValuationReport from a ValuationReportInstruction.
    @Inject
    HKMAValuationReportFunction reportFunc;

    // Function to project an HKMA Dtcc valuation report to ISO20022 format
    @Inject
    Project_HKMADtccValuationReportToIso20022 hkmaDtccValuationReportToIso20022;

    // Function to project an HKMA Tr valuation report to ISO20022 format
    @Inject
    Project_HKMATrValuationReportToIso20022 hkmaTrValuationReportToIso20022;

    /**
     * Demonstrates HKMA valuation reporting with Dtcc ISO-20022 projection.
     * The test projects the report to Dtcc ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void HKMADtccValuationWithISOProjectionExampleTest() throws IOException {

        // Generate the HKMA valuation report
        HKMAValuationReport report = runReport(valuationReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(HKMAValuationReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to Dtcc ISO-20022 format
        projectReport(report, hkmaDtccValuationReportToIso20022::evaluate, Auth030HkmaDtccModelConfig.XML_CONFIG_PATH);
    }

    /**
     * Demonstrates HKMA valuation reporting with Tr ISO-20022 projection.
     * The test projects the report to Tr ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void HKMATrValuationWithISOProjectionExampleTest() throws IOException {

        // Generate the HKMA valuation report
        HKMAValuationReport report = runReport(valuationReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(HKMAValuationReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to Tr ISO-20022 format
        projectReport(report, hkmaTrValuationReportToIso20022::evaluate, Auth030HkmaTrModelConfig.XML_CONFIG_PATH);
    }
}

