package org.isda.drr.example.reporting.collateralmargin;

import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.esma.emir.refit.margin.functions.Project_EsmaEmirMarginReportToIso20022;
import drr.regulation.esma.emir.refit.margin.ESMAEMIRMarginReport;
import drr.regulation.esma.emir.refit.margin.reports.ESMAEMIRMarginReportFunction;
import iso20022.Auth108EsmaModelConfig;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate EU EMIR Refit collateral reporting with ISO-20022 projection.
 * This class validates the transformation of a `CollateralReportInstruction` to an `ESMAEMIRMarginReport`,
 * followed by validation and ISO-20022 projection.
 */
final class EUEMIRRefitCollateralWithISOProjectionTest extends AbstractReportingTest {

    // Function to generate an EMIR Margin Report from a CollateralReportInstruction
    @Inject
    ESMAEMIRMarginReportFunction reportFunc;

    // Function to project an EMIR Margin Report to ISO-20022 format
    @Inject
    Project_EsmaEmirMarginReportToIso20022 emirMarginReportToIso20022;

    /**
     * Demonstrates EU EMIR Refit collateral reporting with ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `CollateralReportInstruction` from JSON.
     * - Generates an `ESMAEMIRMarginReport` and validates it.
     * - Projects the report to ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void EUEMIRCollateralWithISOProjectionExampleTest() throws IOException {

        // Generate the ESMA EMIR margin report
        ESMAEMIRMarginReport report = runReport(collateralReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(ESMAEMIRMarginReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to ISO-20022 format
        projectReport(report, emirMarginReportToIso20022::evaluate, Auth108EsmaModelConfig.XML_CONFIG_PATH);
    }
}
