package org.isda.drr.example.reporting.collateralmargin;

import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.jfsa.rewrite.margin.functions.Project_JFSARewriteMarginReportToIso20022;
import drr.regulation.jfsa.rewrite.margin.JFSAMarginReport;
import drr.regulation.jfsa.rewrite.margin.reports.JFSAMarginReportFunction;
import iso20022.Auth108JfsaModelConfig;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate JFSA collateral reporting with ISO-20022 projection.
 * This class validates the transformation of a `CollateralReportInstruction` to a `JFSAMarginReport`,
 * followed by validation and ISO-20022 projection.
 */
final class JFSACollateralWithISOProjectionTest extends AbstractReportingTest {

    // Function to generate a JFSA Margin Report from a CollateralReportInstruction
    @Inject
    JFSAMarginReportFunction reportFunc;

    // Function to project a JFSA Margin Report to ISO-20022 format
    @Inject
    Project_JFSARewriteMarginReportToIso20022 jfsaMarginReportToIso20022;

    /**
     * Demonstrates JFSA collateral reporting with ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `CollateralReportInstruction` from JSON.
     * - Generates a `JFSAMarginReport` and validates it.
     * - Projects the report to ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void JFSACollateralWithISOProjectionExampleTest() throws IOException {

        // Generate the JFSA margin report
        JFSAMarginReport report = runReport(collateralReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(JFSAMarginReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to ISO-20022 format
        projectReport(report, jfsaMarginReportToIso20022::evaluate, Auth108JfsaModelConfig.XML_CONFIG_PATH);
    }
}
