package org.isda.drr.example.reporting.collateralmargin;

import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.fca.ukemir.refit.margin.functions.Project_FcaUkEmirMarginReportToIso20022;
import drr.regulation.fca.ukemir.refit.margin.FCAUKEMIRMarginReport;
import drr.regulation.fca.ukemir.refit.margin.reports.FCAUKEMIRMarginReportFunction;
import iso20022.Auth108FcaModelConfig;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate UK EMIR collateral reporting with ISO-20022 projection.
 * This class validates the transformation of a `CollateralReportInstruction` to a `FCAUKEMIRMarginReport`,
 * followed by validation and ISO-20022 projection.
 */
final class UKEMIRCollateralWithISOProjectionTest extends AbstractReportingTest {

    // Function to generate a UK EMIR Margin Report from a CollateralReportInstruction
    @Inject
    FCAUKEMIRMarginReportFunction reportFunc;

    // Function to project a UK EMIR Margin Report to ISO-20022 format
    @Inject
    Project_FcaUkEmirMarginReportToIso20022 fcaUkEmirMarginReportToIso20022;

    /**
     * Demonstrates UK EMIR collateral reporting with ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `CollateralReportInstruction` from JSON.
     * - Generates a `FCAUKEMIRMarginReport` and validates it.
     * - Projects the report to ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void UKEMIRCollateralWithISOProjectionExampleTest() throws IOException {

        // Generate the FCA UK EMIR margin report
        FCAUKEMIRMarginReport report = runReport(collateralReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(FCAUKEMIRMarginReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to ISO-20022 format
        projectReport(report, fcaUkEmirMarginReportToIso20022::evaluate, Auth108FcaModelConfig.XML_CONFIG_PATH);
    }
}
