package org.isda.drr.example.reporting.transaction;

import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.esma.emir.refit.trade.functions.Project_EsmaEmirTradeReportToIso20022;
import drr.regulation.esma.emir.refit.trade.ESMAEMIRTransactionReport;
import drr.regulation.esma.emir.refit.trade.reports.ESMAEMIRTradeReportFunction;
import iso20022.Auth030EsmaModelConfig;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate ESMA EMIR transaction reporting with ISO-20022 projection.
 * This class validates the transformation of a `TransactionReportInstruction` to a `ESMATransactionReport`,
 * followed by validation and ISO-20022 projection.
 */
final class EUEMIRRefitTradeWithISOProjectionTest extends AbstractReportingTest {

    // Function to generate an EMIR refit Trade Report from a TransactionReportInstruction
    @Inject
    ESMAEMIRTradeReportFunction reportFunc;

    // Function to project an EMIR refit Trade Report to ISO-20022 format
    @Inject
    Project_EsmaEmirTradeReportToIso20022 emirTradeReportToIso20022;


    /**
     * Demonstrates EMIR REFIT trade reporting with ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `TransactionReportInstruction` from JSON.
     * - Generates an `ESMAEMIRTransactionReport` and validates it.
     * - Projects the report to ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void EUEMIRTradeWithISOProjectionExampleTest() throws IOException {

        // Generate the ESMA EMIR transaction report
        ESMAEMIRTransactionReport report = runReport(transactionReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(ESMAEMIRTransactionReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to ISO-20022 format
        projectReport(report, emirTradeReportToIso20022::evaluate, Auth030EsmaModelConfig.XML_CONFIG_PATH);
    }
}
