package org.isda.drr.example.reporting.valuation;

import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.mas.rewrite.valuation.functions.Project_MASValuationReportToIso20022;
import drr.regulation.mas.rewrite.valuation.MASValuationReport;
import drr.regulation.mas.rewrite.valuation.reports.MASValuationReportFunction;
import iso20022.Auth030MasModelConfig;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate MAS valuation reporting with ISO-20022 projection.
 * This class validates the transformation of a `ValuationReportInstruction` to a `MASValuationReport`,
 * followed by validation and ISO-20022 projection.
 */
final class MASValuationWithISOProjectionTest extends AbstractReportingTest {

    // Function to project a MAS valuation report to ISO-20022 format
    @Inject
    Project_MASValuationReportToIso20022 masValuationReportToIso20022;

    // Function to generate a MAS valuation report from a ValuationReportInstruction
    @Inject
    MASValuationReportFunction reportFunc;


    /**
     * Demonstrates MAS valuation reporting with ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `ValuationReportInstruction` from JSON.
     * - Generates a `MASValuationReport` and validates it.
     * - Projects the report to ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or writing.
     */
    @Test
    void MASValuationWithISOProjectionExampleTest() throws IOException {

        // Generate the MAS valuation report
        MASValuationReport report = runReport(valuationReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(MASValuationReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to ISO-20022 format
        projectReport(report, masValuationReportToIso20022::evaluate, Auth030MasModelConfig.XML_CONFIG_PATH);
    }
}
