package org.isda.drr.example.reporting.collateralmargin;

import iso20022.Auth108HkmaDtccModelConfig;
import iso20022.Auth108HkmaTrModelConfig;
import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.hkma.rewrite.margin.dtcc.functions.Project_HKMADtccMarginReportToIso20022;
import drr.projection.iso20022.hkma.rewrite.margin.tr.functions.Project_HKMATrMarginReportToIso20022;
import drr.regulation.hkma.rewrite.margin.HKMAMarginReport;
import drr.regulation.hkma.rewrite.margin.reports.HKMAMarginReportFunction;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate HKMA collateral reporting with ISO-20022 projection.
 * This class validates the transformation of a `CollateralReportInstruction` to an `HKMAMarginReport`,
 * followed by validation and ISO-20022 projection.
 */
final class HKMACollateralWithISOProjectionTest extends AbstractReportingTest {

    // Function to generate an HKMA Margin Report from a CollateralReportInstruction
    @Inject
    HKMAMarginReportFunction reportFunc;

    // Function to project an HKMA Dtcc margin report to ISO20022 format
    @Inject
    Project_HKMADtccMarginReportToIso20022 hkmaDtccMarginReportToIso20022;

    // Function to project an HKMA Tr margin report to ISO20022 format
    @Inject
    Project_HKMATrMarginReportToIso20022 hkmaTrMarginReportToIso20022;

    /**
     * Demonstrates HKMA margin reporting with Dtcc ISO-20022 projection.
     * Generates an HKMA margin report from a CollateralReportInstruction and validates it
     * The test performs the following steps:
     * - Loads a `CollateralReportInstruction` from JSON.
     * - Generates a `HKMAMarginReport`.
     * - Validates the `HKMAMarginReport`
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void HKMADtccCollateralWithISOProjectionExampleTest() throws IOException {

        // Generate the HKMA margin report
        HKMAMarginReport report = runReport(collateralReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(HKMAMarginReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to Dtcc ISO-20022 format
        projectReport(report, hkmaDtccMarginReportToIso20022::evaluate, Auth108HkmaDtccModelConfig.XML_CONFIG_PATH);
    }

    /**
     * Demonstrates HKMA collateral reporting with Tr ISO-20022 projection.
     * Generates an HKMA margin report from a CollateralReportInstruction and validates it
     * The test performs the following steps:
     * - Loads a `CollateralReportInstruction` from JSON.
     * - Generates a `HKMAMarginReport`.
     * - Validates the `HKMAMarginReport`
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void HKMATrCollateralWithISOProjectionExampleTest() throws IOException {

        // Generate the HKMA margin report
        HKMAMarginReport report = runReport(collateralReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(HKMAMarginReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to Tr ISO-20022 format
        projectReport(report, hkmaTrMarginReportToIso20022::evaluate, Auth108HkmaTrModelConfig.XML_CONFIG_PATH);
    }
}
