package org.isda.drr.example.reporting.transaction;

import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.hkma.rewrite.trade.dtcc.functions.Project_HKMADtccTradeReportToIso20022;
import drr.projection.iso20022.hkma.rewrite.trade.tr.functions.Project_HKMATrTradeReportToIso20022;
import drr.regulation.hkma.rewrite.trade.HKMATransactionReport;
import drr.regulation.hkma.rewrite.trade.reports.HKMATradeReportFunction;
import iso20022.Auth030HkmaDtccModelConfig;
import iso20022.Auth030HkmaTrModelConfig;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate HKMA trade reporting with ISO-20022 projection.
 * This class validates the transformation of a `TransactionReportInstruction` to a `HKMATransactionReport`,
 * followed by validation and ISO-20022 projection.
 */
final class HKMATradeWithISOProjectionTest extends AbstractReportingTest {

    // Function to generate an HKMA Trade Report from a TransactionReportInstruction
    @Inject
    HKMATradeReportFunction reportFunc;

    // Function to project an HKMA Trade Report to Dtcc ISO-20022 format
    @Inject
    Project_HKMADtccTradeReportToIso20022 hkmaDtccTradeReportToIso20022;

    // Function to project an HKMA Trade Report to Tr ISO-20022 format
    @Inject
    Project_HKMATrTradeReportToIso20022 hkmaTrTradeReportToIso20022;

    /**
     * Demonstrates HKMA trade reporting with Dtcc ISO-20022 projection.
     * The test performs the following steps:
     *  - Loads a `TransactionReportInstruction` from JSON.
     *  - Generates a `HKMATransactionReport`.
     * - Projects the report to Dtcc ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void HKMADtccTradeWithISOProjectionExampleTest() throws IOException {

        // Generate the HKMA transaction report
        HKMATransactionReport report = runReport(transactionReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(HKMATransactionReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to Dtcc ISO-20022 format
        projectReport(report, hkmaDtccTradeReportToIso20022::evaluate, Auth030HkmaDtccModelConfig.XML_CONFIG_PATH);
    }

    /**
     * Demonstrates HKMA trade reporting with Tr ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `TransactionReportInstruction` from JSON.
     * - Generates a `HKMATransactionReport`.
     * - Projects the report to Tr ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void HKMATrTradeWithISOProjectionExampleTest() throws IOException {

        // Generate the HKMA transaction report
        HKMATransactionReport report = runReport(transactionReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(HKMATransactionReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to Tr ISO-20022 format
        projectReport(report, hkmaTrTradeReportToIso20022::evaluate, Auth030HkmaTrModelConfig.XML_CONFIG_PATH);
    }
}