package org.isda.drr.example.reporting.transaction;

import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import drr.projection.iso20022.mas.rewrite.trade.functions.Project_MASTradeReportToIso20022;
import drr.regulation.mas.rewrite.trade.MASTransactionReport;
import drr.regulation.mas.rewrite.trade.reports.MASTradeReportFunction;
import iso20022.Auth030MasModelConfig;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate MAS trade reporting with ISO-20022 projection.
 * This class validates the transformation of a `TransactionReportInstruction` to a `MASTransactionReport`,
 * followed by validation and ISO-20022 projection.
 */
final class MASTradeWithISOProjectionTest extends AbstractReportingTest {

    // Function to generate an MAS Trade Report from a TransactionReportInstruction
    @Inject
    MASTradeReportFunction reportFunc;

    // Function to project an MAS Trade Report to ISO-20022 format
    @Inject
    Project_MASTradeReportToIso20022 masTradeReportToIso20022;

    /**
     * Demonstrates MAS trade reporting with ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `TransactionReportInstruction` from JSON.
     * - Generates a `MASTransactionReport` and validates it.
     * - Projects the report to ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or processing.
     */
    @Test
    void MASTradeWithISOProjectionExampleTest() throws IOException {

        // Generate the MAS transaction report
        MASTransactionReport report = runReport(transactionReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(MASTransactionReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to ISO-20022 format
        projectReport(report, masTradeReportToIso20022::evaluate, Auth030MasModelConfig.XML_CONFIG_PATH);
    }
}
