package org.isda.drr.example.reporting.valuation;

import drr.projection.iso20022.esma.emir.refit.valuation.functions.Project_EsmaEmirValuationReportToIso20022;
import drr.regulation.esma.emir.refit.valuation.ESMAValuationReport;
import drr.regulation.esma.emir.refit.valuation.reports.ESMAEMIRValuationReportFunction;
import iso20022.Auth030EsmaModelConfig;
import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate ESMA EMIR valuation reporting with ISO-20022 projection.
 * The test validates the transformation of a `ValuationReportInstruction` to an `ESMAValuationReport`,
 * followed by validation and ISO-20022 projection.
 */
final class EUEMIRValuationWithISOProjectionTest extends AbstractReportingTest {

    // Function to project a ESMA EMIR valuation report to ISO-20022 format
    @Inject
    Project_EsmaEmirValuationReportToIso20022 esmaEmirValuationReportToIso20022;

    // Function to generate an ESMAEMIRValuationReport from a ValuationReportInstruction.
    @Inject
    ESMAEMIRValuationReportFunction reportFunc;


    /**
     * Demonstrates ESMA EMIR valuation reporting with ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `ValuationReportInstruction` from JSON.
     * - Generates an `ESMAEMIRValuationReport` and validates it.
     * - Projects the report to ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or writing.
     */
    @Test
    void EUEMIRValuationWithISOProjectionExampleTest() throws IOException {

        // Generate the ESMA valuation report
        ESMAValuationReport report = runReport(valuationReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(ESMAValuationReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to ISO-20022 format
        projectReport(report, esmaEmirValuationReportToIso20022::evaluate, Auth030EsmaModelConfig.XML_CONFIG_PATH);
    }
}

