package org.isda.drr.example.reporting.valuation;

import drr.projection.iso20022.fca.ukemir.refit.valuation.functions.Project_FcaUkEmirValuationReportToIso20022;
import drr.regulation.fca.ukemir.refit.valuation.FCAValuationReport;
import drr.regulation.fca.ukemir.refit.valuation.reports.FCAUKEMIRValuationReportFunction;
import iso20022.Auth030FcaModelConfig;
import jakarta.inject.Inject;
import com.regnosys.rosetta.common.validation.ValidationReport;
import org.isda.drr.example.AbstractReportingTest;
import org.isda.drr.example.util.ReportingTestUtils;
import org.junit.jupiter.api.Test;

import java.io.IOException;

import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * Test class to demonstrate FCA UK EMIR valuation reporting with ISO-20022 projection.
 * The test validates the transformation of a `ValuationReportInstruction` to an `FCAValuationReport`,
 * followed by validation and ISO-20022 projection.
 */
final class UKEMIRValuationWithISOProjectionTest extends AbstractReportingTest {

    // Function to project an FCA UK EMIR valuation report to ISO-20022 format
    @Inject
    Project_FcaUkEmirValuationReportToIso20022 fcaUkEmirValuationReportToIso20022;

    // Function to generate an FCAUKEMIRValuationReport from a ValuationReportInstruction.
    @Inject
    FCAUKEMIRValuationReportFunction reportFunc;

    /**
     * Demonstrates FCA UK EMIR valuation reporting with ISO-20022 projection.
     * The test performs the following steps:
     * - Loads a `ValuationReportInstruction` from JSON.
     * - Generates an `FCAValuationReport` and validates it.
     * - Projects the report to ISO-20022 format.
     *
     * @throws IOException If there is an error during file reading or writing.
     */
    @Test
    void UKEMIRValuationWithISOProjectionExampleTest() throws IOException {

        // Generate the FCA valuation report
        FCAValuationReport report = runReport(valuationReportInstruction, reportFunc::evaluate);
        assertNotNull(report, "The report is null");

        // Validate the report and print validation results
        ValidationReport validationReport = validator.runProcessStep(FCAValuationReport.class, report);
        ReportingTestUtils.printValidation(validationReport);

        // Project the report to ISO-20022 format
        projectReport(report, fcaUkEmirValuationReportToIso20022::evaluate, Auth030FcaModelConfig.XML_CONFIG_PATH);
    }
}


